/*
#     ___  _ _      ___
#    |    | | |    |
# ___|    |   | ___|    PS2DEV Open Source Project.
#----------------------------------------------------------
# (c) 2006 Eugene Plotnikov <e-plotnikov@operamail.com>
# (c) 2003 Marcus R. Brown <mrbrown@0xd6.org>
# Licenced under Academic Free License version 2.0
# Review ps2sdk README & LICENSE files for further details.
#
*/
#include "SMS.h"
#include "SMS_EE.h"
#include "SMS_IOP.h"
#include "SMS_DMA.h"
#include "SMS_Locale.h"
#include "SMS_GUI.h"
#include "SMS_Data.h"
#include "SMS_Config.h"
#include "SMS_PAD.h"
#include "SMS_RC.h"

#include <loadfile.h>

#include <kernel.h>
#include <malloc.h>
#include <limits.h>
#include <string.h>
#include <fileio.h>

unsigned char* g_pSPRTop;

static char s_pROMVER[] __attribute__(   (  section( ".data" ), aligned( 1 )  )   ) = "rom0:ROMVER";

#define ALIGN( x, a ) (   (  ( x ) + ( a ) - 1  ) & ~(  ( a ) - 1  )   )

unsigned int SMS_Linesize ( unsigned int aWidth, unsigned int* apLinesize ) {

 const int lYWidth = aWidth + 32;

 *apLinesize = ALIGN( lYWidth, 32 ) >> 4;

 return 2;

}  /* end SMS_Linesize */

void* SMS_Realloc ( void* apData, unsigned int* apSize, unsigned int aMinSize ) {

 if ( aMinSize < *apSize ) return apData;

 *apSize = 17 * aMinSize / 16 + 32;

 return realloc ( apData, *apSize );

}  /* SMS_Realloc */

extern long MUL64 ( long, long );

long SMS_Rescale ( long anA, long aB, long aC ){

 int           i;
 unsigned long lA0, lA1, lB0, lB1, lT1, lT1a;
 long          lR;

 if ( anA < 0 ) return -SMS_Rescale ( -anA, aB, aC );

 lR = aC >> 1;

 if ( aB <= INT_MAX && aC <= INT_MAX ) {
  if ( anA <= INT_MAX )
   return ( anA * aB + lR ) / aC;
  else return anA / aC * aB + ( anA % aC * aB + lR ) / aC;
 }  /* end if */

 lA0  = anA & 0xFFFFFFFF;
 lA1  = anA >> 32;
 lB0  = aB  & 0xFFFFFFFF;
 lB1  = aB  >> 32;
 lT1  = MUL64 ( lA0, lB1 ) + MUL64 ( lA1, lB0 );
 lT1a = lT1 << 32;
 lA0  = MUL64 ( lA0, lB0 ) + lT1a;
 lA1  = MUL64 ( lA1, lB1 ) + ( lT1 >> 32 ) + ( lA0 < lT1a );
 lA0 += lR;
 lA1 += lA0 < lR;

 for ( i = 63; i >= 0; --i ) {

  lA1 += lA1 + (  ( lA0 >> i ) & 1  );
  lT1 += lT1;

  if ( aC <= lA1 ){
   lA1 -= aC;
   ++lT1;
  }  /* end if */

 }  /* end for */

 return lT1;

}  /* end SMS_Rescale */

void SMS_Strcat ( char* apStr1, const char* apStr2 ) {

 strcpy (  apStr1 + strlen ( apStr1 ), apStr2  );

}  /* end SMS_Strcat */

int SMS_rand ( void ) {

 static int s_lSeed = 1;

 return s_lSeed = ( 161140751 * s_lSeed + 13 ) % 219441163;

}  /* end SMS_rand */

char* SMS_ReverseString ( char* apStr, int aLen ) {

 char* lpFirst = apStr;
 char* lpLast  = apStr + aLen;

 for ( ; lpFirst != lpLast && lpFirst != --lpLast; ++lpFirst ) {

  char lChr = *lpFirst;

  *lpFirst = *lpLast;
  *lpLast  = lChr;

 }  /* end for */

 return apStr;

}  /* end SMS_ReverseString */

void SMS_EEInit ( void ) {

 unsigned char      lROMVer[ 16 ];
 volatile DMACRegs* lpDMAC = DMAC;
 int                lFD;

 lpDMAC -> m_CTRL = DMA_SET_CTRL( 1, D_CTRL_RELE_ON, D_CTRL_MFD_OFF, D_CTRL_STS_UNSPEC, D_CTRL_STD_OFF, D_CTRL_RCYC_8 );
 lpDMAC -> m_STAT = 0;
 lpDMAC -> m_PCR  = 0;
 lpDMAC -> m_SQWC = 0;
 lpDMAC -> m_RBSR = 0;
 lpDMAC -> m_RBOR = 0;

 ChangeThreadPriority (  GetThreadId (), SMS_THREAD_PRIORITY  );

 lFD = fioOpen ( s_pROMVER, O_RDONLY );
 fioRead (  lFD, lROMVer, sizeof ( lROMVer )  );
 fioClose ( lFD );

 g_pBXDATASYS[ 6 ] = lROMVer[ 4 ] == 'E' ? 'E' : ( lROMVer[ 4 ] == 'J' ? 'I' : 'A');

}  /* end SMS_EEInit */

static unsigned char s_Loader[] __attribute__(   (  aligned( 16 ), section( ".data" )  )   ) = {
	0xc0, 0xff, 0xbd, 0x27, 0x00, 0x01, 0x02, 0x3c, 0x20, 0x00, 0xb0, 0xff, 0x00, 0x1d, 0x42, 0x24, 
	0x30, 0x00, 0xbf, 0xff, 0x2d, 0x80, 0xa0, 0x00, 0x2d, 0x30, 0x40, 0x00, 0x00, 0x00, 0x83, 0x80, 
	0x10, 0x00, 0xa2, 0xaf, 0x0a, 0x00, 0x60, 0x10, 0x00, 0x00, 0x85, 0x90, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0xc5, 0xa0, 0x01, 0x00, 0x84, 0x24, 0x01, 0x00, 0xc6, 0x24, 0x00, 0x00, 0x82, 0x80, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf9, 0xff, 0x40, 0x14, 0x00, 0x00, 0x85, 0x90, 
	0x00, 0x00, 0xc0, 0xa0, 0x00, 0x01, 0x04, 0x3c, 0x2d, 0x28, 0xa0, 0x03, 0x8c, 0x01, 0x40, 0x0c, 
	0x00, 0x1d, 0x84, 0x24, 0x00, 0x00, 0xa3, 0x8f, 0x07, 0x00, 0x60, 0x14, 0x00, 0x00, 0x00, 0x00, 
	0x09, 0xf8, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0xb0, 0xdf, 0x30, 0x00, 0xbf, 0xdf, 
	0x08, 0x00, 0xe0, 0x03, 0x40, 0x00, 0xbd, 0x27, 0x78, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 
	0x62, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x98, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 
	0x46, 0x00, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 0xfd, 0xff, 0x40, 0x10, 0x00, 0x01, 0x04, 0x3c, 
	0x2d, 0x28, 0x00, 0x00, 0x2d, 0x30, 0x00, 0x00, 0x84, 0x01, 0x40, 0x0c, 0x00, 0x1c, 0x84, 0x24, 
	0x00, 0x01, 0x04, 0x3c, 0x2d, 0x30, 0x00, 0x00, 0x2d, 0x28, 0x00, 0x00, 0x84, 0x01, 0x40, 0x0c, 
	0x10, 0x1c, 0x84, 0x24, 0x00, 0x01, 0x04, 0x3c, 0x2d, 0x30, 0x00, 0x00, 0x2d, 0x28, 0x00, 0x00, 
	0x84, 0x01, 0x40, 0x0c, 0x20, 0x1c, 0x84, 0x24, 0x54, 0x00, 0x40, 0x0c, 0x2d, 0x20, 0x00, 0x00, 
	0x54, 0x00, 0x40, 0x0c, 0x02, 0x00, 0x04, 0x24, 0x04, 0x00, 0xa5, 0x8f, 0x00, 0x00, 0xa4, 0x8f, 
	0x10, 0x00, 0xa7, 0x27, 0x50, 0x00, 0x40, 0x0c, 0x01, 0x00, 0x06, 0x24, 0xd8, 0xff, 0x00, 0x10, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xff, 0xbd, 0x27, 0x00, 0x00, 0xbf, 0xff, 
	0x58, 0x00, 0x40, 0x0c, 0x04, 0x00, 0x04, 0x24, 0x04, 0x00, 0x03, 0x3c, 0x00, 0x00, 0xbf, 0xdf, 
	0x24, 0x10, 0x43, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x10, 0x00, 0xbd, 0x27, 0x00, 0x00, 0x00, 0x00, 
	0x07, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x64, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x7a, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x10, 0x00, 0x82, 0x8c, 0xfe, 0xff, 0x03, 0x24, 0x18, 0x00, 0x80, 0xac, 0x24, 0x10, 0x43, 0x00, 
	0x08, 0x00, 0xe0, 0x03, 0x10, 0x00, 0x82, 0xac, 0xe0, 0xff, 0xbd, 0x27, 0x00, 0x80, 0x02, 0x3c, 
	0x10, 0x00, 0xbf, 0xff, 0x00, 0x80, 0x05, 0x3c, 0x00, 0x00, 0xb0, 0xff, 0x0a, 0x00, 0x42, 0x34, 
	0x09, 0x00, 0xa5, 0x34, 0x20, 0x00, 0x83, 0x8c, 0x15, 0x00, 0x62, 0x10, 0x1c, 0x00, 0x90, 0x8c, 
	0x0d, 0x00, 0x65, 0x50, 0x28, 0x00, 0x82, 0x8c, 0x08, 0x00, 0x02, 0x8e, 0x03, 0x00, 0x40, 0x04, 
	0x2d, 0x20, 0x40, 0x00, 0xdc, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x40, 0x0c, 
	0x00, 0x00, 0x04, 0x8e, 0x10, 0x00, 0xbf, 0xdf, 0x00, 0x00, 0x00, 0xae, 0x00, 0x00, 0xb0, 0xdf, 
	0x08, 0x00, 0xe0, 0x03, 0x20, 0x00, 0xbd, 0x27, 0x24, 0x00, 0x83, 0x8c, 0x14, 0x00, 0x02, 0xae, 
	0x24, 0x00, 0x03, 0xae, 0x2c, 0x00, 0x82, 0x8c, 0xef, 0xff, 0x00, 0x10, 0x18, 0x00, 0x02, 0xae, 
	0x1c, 0x00, 0x02, 0x8e, 0xed, 0xff, 0x40, 0x50, 0x08, 0x00, 0x02, 0x8e, 0x09, 0xf8, 0x40, 0x00, 
	0x20, 0x00, 0x04, 0x8e, 0xe9, 0xff, 0x00, 0x10, 0x08, 0x00, 0x02, 0x8e, 0x00, 0x00, 0x00, 0x00, 
	0x28, 0x00, 0xa6, 0x8c, 0x0e, 0x00, 0xc0, 0x10, 0x2d, 0x28, 0x00, 0x00, 0x08, 0x00, 0xc3, 0x8c, 
	0x08, 0x00, 0x60, 0x50, 0x14, 0x00, 0xc6, 0x8c, 0x00, 0x00, 0x62, 0x8c, 0x08, 0x00, 0x44, 0x10, 
	0x2d, 0x28, 0x60, 0x00, 0x38, 0x00, 0x63, 0x8c, 0xfc, 0xff, 0x60, 0x54, 0x00, 0x00, 0x62, 0x8c, 
	0x14, 0x00, 0xc6, 0x8c, 0xf6, 0xff, 0xc0, 0x54, 0x08, 0x00, 0xc3, 0x8c, 0x2d, 0x28, 0x00, 0x00, 
	0x08, 0x00, 0xe0, 0x03, 0x2d, 0x10, 0xa0, 0x00, 0xc0, 0xff, 0xbd, 0x27, 0x00, 0x00, 0xb0, 0xff, 
	0x2d, 0x80, 0x80, 0x00, 0x2d, 0x20, 0xa0, 0x00, 0x20, 0x00, 0xb2, 0xff, 0x10, 0x00, 0xb1, 0xff, 
	0x30, 0x00, 0xbf, 0xff, 0x6c, 0x01, 0x40, 0x0c, 0x2d, 0x88, 0xa0, 0x00, 0x2d, 0x90, 0x40, 0x00, 
	0x14, 0x00, 0x04, 0x8e, 0x00, 0x80, 0x02, 0x3c, 0x1c, 0x00, 0x03, 0x8e, 0x09, 0x00, 0x42, 0x34, 
	0x14, 0x00, 0x44, 0xae, 0x20, 0x00, 0x42, 0xae, 0x2d, 0x28, 0x20, 0x02, 0x1c, 0x00, 0x43, 0xae, 
	0x88, 0x00, 0x40, 0x0c, 0x20, 0x00, 0x04, 0x8e, 0x15, 0x00, 0x40, 0x50, 0x24, 0x00, 0x40, 0xae, 
	0x08, 0x00, 0x43, 0x8c, 0x24, 0x00, 0x42, 0xae, 0x28, 0x00, 0x43, 0xae, 0x14, 0x00, 0x42, 0x8c, 
	0x2c, 0x00, 0x42, 0xae, 0x00, 0x80, 0x04, 0x3c, 0x2d, 0x28, 0x40, 0x02, 0x08, 0x00, 0x84, 0x34, 
	0x40, 0x00, 0x06, 0x24, 0x2d, 0x38, 0x00, 0x00, 0x2d, 0x40, 0x00, 0x00, 0x6c, 0x02, 0x40, 0x0c, 
	0x2d, 0x48, 0x00, 0x00, 0x00, 0x00, 0xb0, 0xdf, 0x30, 0x00, 0xbf, 0xdf, 0x20, 0x00, 0xb2, 0xdf, 
	0x10, 0x00, 0xb1, 0xdf, 0x08, 0x00, 0xe0, 0x03, 0x40, 0x00, 0xbd, 0x27, 0x00, 0x00, 0x00, 0x00, 
	0x28, 0x00, 0x40, 0xae, 0xef, 0xff, 0x00, 0x10, 0x2c, 0x00, 0x40, 0xae, 0x00, 0x00, 0x00, 0x00, 
	0xf0, 0xff, 0xbd, 0x27, 0x00, 0x00, 0xbf, 0xff, 0x34, 0x00, 0x86, 0x8c, 0x40, 0x00, 0xc8, 0x8c, 
	0x0c, 0x00, 0x02, 0x8d, 0x20, 0x00, 0x40, 0x10, 0x2d, 0x38, 0x80, 0x00, 0x10, 0x00, 0x02, 0x8d, 
	0x38, 0x00, 0x46, 0xac, 0x20, 0x00, 0xe2, 0x8c, 0x14, 0x00, 0xe3, 0x8c, 0x24, 0x00, 0xc2, 0xac, 
	0x20, 0x00, 0xc3, 0xac, 0x24, 0x00, 0xe2, 0x8c, 0x28, 0x00, 0xe4, 0x8c, 0x0c, 0x00, 0xc2, 0xac, 
	0x1c, 0x00, 0xe5, 0x8c, 0x2c, 0x00, 0xe3, 0x8c, 0x28, 0x00, 0xc4, 0xac, 0x2c, 0x00, 0xc3, 0xac, 
	0x10, 0x00, 0x06, 0xad, 0x30, 0x00, 0xe2, 0x8c, 0x1c, 0x00, 0xc5, 0xac, 0x30, 0x00, 0xc2, 0xac, 
	0x10, 0x00, 0xe3, 0x8c, 0x34, 0x00, 0xc3, 0xac, 0x00, 0x00, 0x04, 0x8d, 0x04, 0x00, 0x80, 0x04, 
	0x00, 0x00, 0xbf, 0xdf, 0x04, 0x00, 0x02, 0x8d, 0x03, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 
	0x08, 0x00, 0xe0, 0x03, 0x10, 0x00, 0xbd, 0x27, 0xd8, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 
	0xfb, 0xff, 0x00, 0x10, 0x00, 0x00, 0xbf, 0xdf, 0xe2, 0xff, 0x00, 0x10, 0x0c, 0x00, 0x06, 0xad, 
	0xe0, 0xff, 0xbd, 0x27, 0x00, 0x00, 0xb0, 0xff, 0x2d, 0x80, 0x80, 0x00, 0x10, 0x00, 0xbf, 0xff, 
	0x6c, 0x01, 0x40, 0x0c, 0x2d, 0x20, 0xa0, 0x00, 0x1c, 0x00, 0x07, 0x8e, 0x14, 0x00, 0x04, 0x8e, 
	0x00, 0x80, 0x03, 0x3c, 0x0c, 0x00, 0x63, 0x34, 0x1c, 0x00, 0x47, 0xac, 0x14, 0x00, 0x44, 0xac, 
	0x40, 0x00, 0x06, 0x24, 0x20, 0x00, 0x43, 0xac, 0x00, 0x80, 0x04, 0x3c, 0x20, 0x00, 0x07, 0x8e, 
	0x2d, 0x28, 0x40, 0x00, 0x28, 0x00, 0x09, 0x8e, 0x08, 0x00, 0x84, 0x34, 0x6c, 0x02, 0x40, 0x0c, 
	0x24, 0x00, 0x08, 0x8e, 0x00, 0x00, 0xb0, 0xdf, 0x10, 0x00, 0xbf, 0xdf, 0x08, 0x00, 0xe0, 0x03, 
	0x20, 0x00, 0xbd, 0x27, 0x00, 0x00, 0x00, 0x00, 0xd0, 0xff, 0xbd, 0x27, 0x00, 0x01, 0x04, 0x3c, 
	0x20, 0x00, 0xbf, 0xff, 0x00, 0x01, 0x02, 0x3c, 0x10, 0x00, 0xb1, 0xff, 0x00, 0x00, 0xb0, 0xff, 
	0xb8, 0x1c, 0x42, 0x8c, 0xb0, 0x1c, 0x83, 0x8c, 0x03, 0x00, 0x62, 0x10, 0x00, 0x01, 0x03, 0x3c, 
	0xb0, 0x1c, 0x82, 0xac, 0xac, 0x1c, 0x60, 0xac, 0xac, 0x1c, 0x62, 0x8c, 0x06, 0x00, 0x40, 0x10, 
	0x00, 0x01, 0x10, 0x3c, 0x20, 0x00, 0xbf, 0xdf, 0x10, 0x00, 0xb1, 0xdf, 0x00, 0x00, 0xb0, 0xdf, 
	0x08, 0x00, 0xe0, 0x03, 0x30, 0x00, 0xbd, 0x27, 0x01, 0x00, 0x11, 0x24, 0x80, 0x1c, 0x10, 0x26, 
	0x88, 0x02, 0x40, 0x0c, 0xac, 0x1c, 0x71, 0xac, 0xbe, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 
	0x2d, 0x30, 0x00, 0x02, 0x04, 0x00, 0x08, 0x8e, 0x00, 0x20, 0x02, 0x3c, 0x14, 0x00, 0x07, 0x8e, 
	0x00, 0x80, 0x04, 0x3c, 0x1c, 0x00, 0x03, 0x8e, 0x25, 0x40, 0x02, 0x01, 0x25, 0x38, 0xe2, 0x00, 
	0x04, 0x00, 0x08, 0xae, 0x25, 0x18, 0x62, 0x00, 0x14, 0x00, 0x07, 0xae, 0x1c, 0x00, 0x03, 0xae, 
	0x00, 0x01, 0x05, 0x3c, 0x88, 0x01, 0xa5, 0x24, 0x26, 0x03, 0x40, 0x0c, 0x08, 0x00, 0x84, 0x34, 
	0x2d, 0x30, 0x00, 0x02, 0x00, 0x80, 0x04, 0x3c, 0x00, 0x01, 0x05, 0x3c, 0x09, 0x00, 0x84, 0x34, 
	0x26, 0x03, 0x40, 0x0c, 0x68, 0x02, 0xa5, 0x24, 0x00, 0x01, 0x05, 0x3c, 0x00, 0x80, 0x04, 0x3c, 
	0x2d, 0x30, 0x00, 0x02, 0x0a, 0x00, 0x84, 0x34, 0x26, 0x03, 0x40, 0x0c, 0x20, 0x03, 0xa5, 0x24, 
	0x00, 0x01, 0x05, 0x3c, 0x00, 0x80, 0x04, 0x3c, 0x2d, 0x30, 0x00, 0x02, 0x0c, 0x00, 0x84, 0x34, 
	0x26, 0x03, 0x40, 0x0c, 0xc0, 0x03, 0xa5, 0x24, 0xd2, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x80, 0x04, 0x3c, 0x58, 0x00, 0x40, 0x0c, 0x02, 0x00, 0x84, 0x34, 0xce, 0xff, 0x40, 0x14, 
	0x20, 0x00, 0xbf, 0xdf, 0x00, 0x01, 0x02, 0x3c, 0x00, 0x80, 0x04, 0x3c, 0x40, 0x21, 0x42, 0x24, 
	0x02, 0x00, 0x84, 0x34, 0x0c, 0x00, 0x51, 0xac, 0x2d, 0x28, 0x40, 0x00, 0x10, 0x00, 0x06, 0x24, 
	0x2d, 0x38, 0x00, 0x00, 0x2d, 0x40, 0x00, 0x00, 0x5c, 0x02, 0x40, 0x0c, 0x2d, 0x48, 0x00, 0x00, 
	0x34, 0x03, 0x40, 0x0c, 0x2d, 0x20, 0x00, 0x00, 0xfd, 0xff, 0x40, 0x10, 0x00, 0x80, 0x04, 0x3c, 
	0x01, 0x00, 0x05, 0x24, 0xe0, 0x01, 0x40, 0x0c, 0x02, 0x00, 0x84, 0x34, 0xba, 0xff, 0x00, 0x10, 
	0x20, 0x00, 0xbf, 0xdf, 0x00, 0x00, 0x00, 0x00, 0xf0, 0xff, 0xbd, 0x27, 0x00, 0x00, 0xbf, 0xff, 
	0x18, 0x03, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xdf, 0x00, 0x01, 0x02, 0x3c, 
	0xac, 0x1c, 0x40, 0xac, 0x08, 0x00, 0xe0, 0x03, 0x10, 0x00, 0xbd, 0x27, 0x00, 0x00, 0x00, 0x00, 
	0x24, 0x00, 0x85, 0x8c, 0x18, 0x00, 0x82, 0x8c, 0x1a, 0x00, 0xa2, 0x00, 0x01, 0x00, 0x40, 0x50, 
	0xcd, 0x01, 0x00, 0x00, 0x14, 0x00, 0x82, 0x8c, 0x10, 0x18, 0x00, 0x00, 0x80, 0x29, 0x03, 0x00, 
	0x01, 0x00, 0x63, 0x24, 0x21, 0x10, 0x45, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x24, 0x00, 0x83, 0xac, 
	0x00, 0x01, 0x04, 0x3c, 0xf0, 0xff, 0xbd, 0x27, 0x2d, 0x28, 0x00, 0x00, 0x80, 0x3c, 0x84, 0x24, 
	0x28, 0x00, 0x06, 0x24, 0x00, 0x01, 0x02, 0x3c, 0x00, 0x00, 0xbf, 0xff, 0xe4, 0x01, 0x40, 0x0c, 
	0xf0, 0x1c, 0x40, 0xac, 0x00, 0x00, 0xbf, 0xdf, 0x08, 0x00, 0xe0, 0x03, 0x10, 0x00, 0xbd, 0x27, 
	0xf0, 0xff, 0xbd, 0x27, 0x2d, 0x38, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xff, 0x72, 0x03, 0x40, 0x0c, 
	0x2d, 0x40, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xdf, 0x08, 0x00, 0xe0, 0x03, 0x10, 0x00, 0xbd, 0x27, 
	0xe0, 0xff, 0xbd, 0x27, 0x6c, 0x00, 0x02, 0x3c, 0x2d, 0x30, 0xa0, 0x00, 0x61, 0x6c, 0x42, 0x34, 
	0x2d, 0x28, 0xa0, 0x03, 0x01, 0x00, 0x07, 0x24, 0x10, 0x00, 0xbf, 0xff, 0xb4, 0x03, 0x40, 0x0c, 
	0x00, 0x00, 0xa2, 0xaf, 0x10, 0x00, 0xbf, 0xdf, 0x08, 0x00, 0xe0, 0x03, 0x20, 0x00, 0xbd, 0x27, 
	0xe0, 0xff, 0xbd, 0x27, 0x10, 0x00, 0xbf, 0xff, 0x62, 0x01, 0x40, 0x0c, 0x00, 0x00, 0xb0, 0xff, 
	0x2d, 0x28, 0x00, 0x00, 0xea, 0x03, 0x40, 0x0c, 0x2d, 0x20, 0x00, 0x00, 0x06, 0x00, 0x40, 0x14, 
	0x2d, 0x80, 0x40, 0x00, 0x2d, 0x10, 0x00, 0x02, 0x10, 0x00, 0xbf, 0xdf, 0x00, 0x00, 0xb0, 0xdf, 
	0x08, 0x00, 0xe0, 0x03, 0x20, 0x00, 0xbd, 0x27, 0xac, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 
	0xfd, 0xff, 0x40, 0x10, 0x2d, 0x10, 0x00, 0x02, 0xf8, 0xff, 0x00, 0x10, 0x10, 0x00, 0xbf, 0xdf, 
	0xf0, 0xff, 0xbd, 0x27, 0x00, 0x00, 0xbf, 0xff, 0x58, 0x00, 0x40, 0x0c, 0x04, 0x00, 0x04, 0x24, 
	0x04, 0x00, 0x03, 0x3c, 0x04, 0x00, 0x04, 0x24, 0x24, 0x10, 0x43, 0x00, 0x04, 0x00, 0x05, 0x3c, 
	0x05, 0x00, 0x40, 0x14, 0x2d, 0x18, 0x00, 0x00, 0x00, 0x00, 0xbf, 0xdf, 0x2d, 0x10, 0x60, 0x00, 
	0x08, 0x00, 0xe0, 0x03, 0x10, 0x00, 0xbd, 0x27, 0xe0, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 
	0xf9, 0xff, 0x00, 0x10, 0x01, 0x00, 0x03, 0x24, 0x00, 0x60, 0x03, 0x40, 0x01, 0x00, 0x02, 0x3c, 
	0x24, 0x18, 0x62, 0x00, 0x2d, 0x20, 0x00, 0x00, 0x0c, 0x00, 0x60, 0x10, 0x2b, 0x28, 0x03, 0x00, 
	0x01, 0x00, 0x02, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x39, 0x00, 0x00, 0x42, 0x0f, 0x04, 0x00, 0x00, 
	0x00, 0x60, 0x03, 0x40, 0x00, 0x00, 0x00, 0x00, 0x24, 0x18, 0x62, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0xf9, 0xff, 0x60, 0x14, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x20, 0xa0, 0x00, 0x08, 0x00, 0xe0, 0x03, 
	0x2d, 0x10, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x02, 0x40, 0x01, 0x00, 0x03, 0x3c, 
	0x24, 0x10, 0x43, 0x00, 0x38, 0x00, 0x00, 0x42, 0x08, 0x00, 0xe0, 0x03, 0x2b, 0x10, 0x02, 0x00, 
	0xcc, 0xff, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0xbd, 0xff, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x79, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x08, 0x00, 0xc2, 0x2c, 0x1c, 0x00, 0x40, 0x14, 0x2d, 0x18, 0x80, 0x00, 0x0f, 0x00, 0x82, 0x30, 
	0x19, 0x00, 0x40, 0x14, 0x2d, 0x38, 0x80, 0x00, 0xff, 0x00, 0xa9, 0x30, 0x20, 0x00, 0xca, 0x2c, 
	0x2d, 0x40, 0x20, 0x01, 0x38, 0x1a, 0x08, 0x00, 0x25, 0x40, 0x69, 0x00, 0xe9, 0x1e, 0x08, 0x70, 
	0x0e, 0x00, 0x40, 0x15, 0x08, 0x00, 0xc2, 0x2c, 0x89, 0x43, 0x63, 0x70, 0x00, 0x00, 0xe8, 0x7c, 
	0xe0, 0xff, 0xc6, 0x24, 0x10, 0x00, 0xe7, 0x24, 0x20, 0x00, 0xc2, 0x2c, 0x00, 0x00, 0xe8, 0x7c, 
	0xfa, 0xff, 0x40, 0x10, 0x10, 0x00, 0xe7, 0x24, 0x04, 0x00, 0x00, 0x10, 0x08, 0x00, 0xc2, 0x2c, 
	0xf8, 0xff, 0xc6, 0x24, 0x08, 0x00, 0xe7, 0x24, 0x08, 0x00, 0xc2, 0x2c, 0xfc, 0xff, 0x40, 0x50, 
	0x00, 0x00, 0xe3, 0xfc, 0x2d, 0x18, 0xe0, 0x00, 0xff, 0xff, 0x02, 0x3c, 0xff, 0xff, 0xc6, 0x24, 
	0xff, 0xff, 0x42, 0x34, 0x07, 0x00, 0xc2, 0x10, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x02, 0x3c, 
	0xff, 0xff, 0x42, 0x34, 0x00, 0x00, 0x65, 0xa0, 0xff, 0xff, 0xc6, 0x24, 0xfd, 0xff, 0xc2, 0x14, 
	0x01, 0x00, 0x63, 0x24, 0x08, 0x00, 0xe0, 0x03, 0x2d, 0x10, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x60, 0xff, 0xbd, 0x27, 0x2d, 0x18, 0x00, 0x00, 0x20, 0x00, 0xb0, 0xff, 0xff, 0x00, 0xf0, 0x30, 
	0x71, 0x00, 0x02, 0x2e, 0x80, 0x00, 0xb6, 0xff, 0x70, 0x00, 0xb5, 0xff, 0x2d, 0xb0, 0xa0, 0x00, 
	0x60, 0x00, 0xb4, 0xff, 0x2d, 0xa8, 0x00, 0x01, 0x50, 0x00, 0xb3, 0xff, 0x2d, 0xa0, 0x20, 0x01, 
	0x40, 0x00, 0xb2, 0xff, 0x2d, 0x98, 0x00, 0x00, 0x30, 0x00, 0xb1, 0xff, 0x2d, 0x90, 0x40, 0x01, 
	0x90, 0x00, 0xbf, 0xff, 0x26, 0x00, 0x40, 0x10, 0x2d, 0x88, 0xc0, 0x00, 0x00, 0x12, 0x0a, 0x00, 
	0x04, 0x00, 0xa3, 0x30, 0x25, 0x10, 0x02, 0x02, 0x08, 0x00, 0xc4, 0xac, 0x04, 0x00, 0xc0, 0xac, 
	0x0b, 0x00, 0x40, 0x19, 0x00, 0x00, 0xd0, 0xac, 0x2d, 0x20, 0x00, 0x01, 0x2d, 0x28, 0x40, 0x01, 
	0x00, 0x00, 0xc2, 0xac, 0x01, 0x00, 0x13, 0x24, 0x29, 0x00, 0x60, 0x14, 0x04, 0x00, 0x29, 0xae, 
	0x00, 0x00, 0xb5, 0xaf, 0x04, 0x00, 0xb4, 0xaf, 0x08, 0x00, 0xb2, 0xaf, 0x0c, 0x00, 0xa0, 0xaf, 
	0x00, 0x19, 0x13, 0x00, 0x44, 0x00, 0x02, 0x24, 0x21, 0x18, 0x7d, 0x00, 0x2d, 0x20, 0x20, 0x02, 
	0x0c, 0x00, 0x62, 0xac, 0x2d, 0x28, 0x00, 0x02, 0x00, 0x01, 0x02, 0x3c, 0x08, 0x00, 0x70, 0xac, 
	0xc8, 0x1c, 0x46, 0x8c, 0x01, 0x00, 0x73, 0x26, 0x00, 0x00, 0x71, 0xac, 0x94, 0x04, 0x40, 0x0c, 
	0x04, 0x00, 0x66, 0xac, 0x01, 0x00, 0xc2, 0x32, 0x2d, 0x28, 0x60, 0x02, 0x0f, 0x00, 0x40, 0x10, 
	0x2d, 0x20, 0xa0, 0x03, 0x8c, 0x04, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x18, 0x40, 0x00, 
	0x90, 0x00, 0xbf, 0xdf, 0x2d, 0x10, 0x60, 0x00, 0x80, 0x00, 0xb6, 0xdf, 0x70, 0x00, 0xb5, 0xdf, 
	0x60, 0x00, 0xb4, 0xdf, 0x50, 0x00, 0xb3, 0xdf, 0x40, 0x00, 0xb2, 0xdf, 0x30, 0x00, 0xb1, 0xdf, 
	0x20, 0x00, 0xb0, 0xdf, 0x08, 0x00, 0xe0, 0x03, 0xa0, 0x00, 0xbd, 0x27, 0x88, 0x04, 0x40, 0x0c, 
	0x2d, 0x20, 0xa0, 0x03, 0xf2, 0xff, 0x00, 0x10, 0x2d, 0x18, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x94, 0x04, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 0xd6, 0xff, 0x00, 0x10, 0x00, 0x00, 0xb5, 0xaf, 
	0x2d, 0x18, 0xe0, 0x00, 0x2d, 0x58, 0x00, 0x01, 0x2d, 0x10, 0xc0, 0x00, 0xf0, 0xff, 0xbd, 0x27, 
	0x2d, 0x50, 0x20, 0x01, 0x2d, 0x30, 0xa0, 0x00, 0x2d, 0x38, 0x40, 0x00, 0x2d, 0x40, 0x60, 0x00, 
	0x2d, 0x48, 0x60, 0x01, 0x00, 0x00, 0xbf, 0xff, 0x10, 0x02, 0x40, 0x0c, 0x2d, 0x28, 0x00, 0x00, 
	0x00, 0x00, 0xbf, 0xdf, 0x08, 0x00, 0xe0, 0x03, 0x10, 0x00, 0xbd, 0x27, 0x00, 0x00, 0x00, 0x00, 
	0x2d, 0x18, 0xe0, 0x00, 0x2d, 0x58, 0x00, 0x01, 0x2d, 0x10, 0xc0, 0x00, 0xf0, 0xff, 0xbd, 0x27, 
	0x2d, 0x50, 0x20, 0x01, 0x2d, 0x30, 0xa0, 0x00, 0x2d, 0x38, 0x40, 0x00, 0x2d, 0x40, 0x60, 0x00, 
	0x2d, 0x48, 0x60, 0x01, 0x00, 0x00, 0xbf, 0xff, 0x10, 0x02, 0x40, 0x0c, 0x01, 0x00, 0x05, 0x24, 
	0x00, 0x00, 0xbf, 0xdf, 0x08, 0x00, 0xe0, 0x03, 0x10, 0x00, 0xbd, 0x27, 0x00, 0x00, 0x00, 0x00, 
	0x10, 0x00, 0x82, 0x8c, 0x08, 0x00, 0xe0, 0x03, 0x08, 0x00, 0xa2, 0xac, 0x00, 0x00, 0x00, 0x00, 
	0x10, 0x00, 0x82, 0x8c, 0x1c, 0x00, 0xa6, 0x8c, 0x80, 0x10, 0x02, 0x00, 0x14, 0x00, 0x83, 0x8c, 
	0x21, 0x10, 0x46, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x43, 0xac, 0x00, 0x00, 0x00, 0x00, 
	0xa0, 0xff, 0xbd, 0x27, 0x00, 0x01, 0x04, 0x3c, 0x50, 0x00, 0xbf, 0xff, 0x00, 0x01, 0x02, 0x3c, 
	0x40, 0x00, 0xb2, 0xff, 0x30, 0x00, 0xb1, 0xff, 0x20, 0x00, 0xb0, 0xff, 0xb8, 0x1c, 0x42, 0x8c, 
	0xe8, 0x1c, 0x83, 0x8c, 0x03, 0x00, 0x62, 0x10, 0x00, 0x01, 0x11, 0x3c, 0xe8, 0x1c, 0x82, 0xac, 
	0xe0, 0x1c, 0x20, 0xae, 0xe0, 0x1c, 0x22, 0x8e, 0x06, 0x00, 0x40, 0x10, 0x50, 0x00, 0xbf, 0xdf, 
	0x40, 0x00, 0xb2, 0xdf, 0x30, 0x00, 0xb1, 0xdf, 0x20, 0x00, 0xb0, 0xdf, 0x08, 0x00, 0xe0, 0x03, 
	0x60, 0x00, 0xbd, 0x27, 0xbe, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x06, 0x3c, 
	0x00, 0x01, 0x05, 0x3c, 0x1f, 0x00, 0x08, 0x24, 0xc0, 0x1c, 0xa4, 0x24, 0xc0, 0x1c, 0xa2, 0x8c, 
	0x04, 0x00, 0x83, 0x8c, 0x25, 0x10, 0x46, 0x00, 0x0c, 0x00, 0x87, 0x8c, 0x25, 0x18, 0x66, 0x00, 
	0xc0, 0x1c, 0xa2, 0xac, 0x04, 0x00, 0x83, 0xac, 0xff, 0xff, 0x08, 0x25, 0x00, 0x00, 0xe0, 0xac, 
	0x04, 0x00, 0xe0, 0xac, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0xf9, 0xff, 0x01, 0x05, 0x08, 0x00, 0xe7, 0x24, 0x00, 0x01, 0x02, 0x3c, 0x1f, 0x00, 0x08, 0x24, 
	0xdc, 0x1c, 0x42, 0x8c, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x08, 0x25, 0x00, 0x00, 0x40, 0xac, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0xf9, 0xff, 0x01, 0x05, 0x04, 0x00, 0x42, 0x24, 0x00, 0x01, 0x12, 0x3c, 0x00, 0x01, 0x02, 0x3c, 
	0xc0, 0x1c, 0x50, 0x26, 0x00, 0x01, 0x03, 0x3c, 0x0c, 0x00, 0x04, 0x8e, 0x00, 0x0a, 0x63, 0x24, 
	0xf0, 0x09, 0x42, 0x24, 0x08, 0x00, 0x83, 0xac, 0x00, 0x00, 0x82, 0xac, 0x0c, 0x00, 0x90, 0xac, 
	0xd2, 0x01, 0x40, 0x0c, 0x04, 0x00, 0x90, 0xac, 0x54, 0x00, 0x40, 0x0c, 0x2d, 0x20, 0x00, 0x00, 
	0x00, 0x10, 0x03, 0x3c, 0x10, 0xe0, 0x63, 0x34, 0x00, 0x00, 0x62, 0x8c, 0x20, 0x00, 0x42, 0x30, 
	0x05, 0x00, 0x40, 0x10, 0x00, 0x10, 0x03, 0x3c, 0x20, 0x00, 0x02, 0x24, 0x01, 0x10, 0x01, 0x3c, 
	0x10, 0xe0, 0x22, 0xac, 0x00, 0x10, 0x03, 0x3c, 0x00, 0xc0, 0x63, 0x34, 0x00, 0x00, 0x62, 0x8c, 
	0x00, 0x01, 0x42, 0x30, 0x39, 0x00, 0x40, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x05, 0x3c, 
	0x2d, 0x30, 0x00, 0x00, 0xa0, 0x16, 0xa5, 0x24, 0x74, 0x04, 0x40, 0x0c, 0x05, 0x00, 0x04, 0x24, 
	0x00, 0x01, 0x03, 0x3c, 0x05, 0x00, 0x04, 0x24, 0x36, 0x04, 0x40, 0x0c, 0xe4, 0x1c, 0x62, 0xac, 
	0x01, 0x00, 0x02, 0x24, 0x00, 0x80, 0x04, 0x3c, 0x58, 0x00, 0x40, 0x0c, 0xe0, 0x1c, 0x22, 0xae, 
	0x21, 0x00, 0x40, 0x14, 0x08, 0x00, 0x02, 0xae, 0x02, 0x00, 0x10, 0x3c, 0x00, 0x00, 0x00, 0x00, 
	0x58, 0x00, 0x40, 0x0c, 0x04, 0x00, 0x04, 0x24, 0x24, 0x10, 0x50, 0x00, 0xfc, 0xff, 0x40, 0x10, 
	0x02, 0x00, 0x04, 0x24, 0x58, 0x00, 0x40, 0x0c, 0x00, 0x01, 0x11, 0x3c, 0xc0, 0x1c, 0x30, 0x26, 
	0x08, 0x00, 0x02, 0xae, 0x2d, 0x28, 0x40, 0x00, 0xe0, 0x01, 0x40, 0x0c, 0x00, 0x80, 0x04, 0x3c, 
	0x2d, 0x28, 0x00, 0x02, 0x00, 0x80, 0x04, 0x3c, 0xe0, 0x01, 0x40, 0x0c, 0x01, 0x00, 0x84, 0x34, 
	0x0c, 0x00, 0xa0, 0xaf, 0xc0, 0x1c, 0x22, 0x8e, 0x00, 0x80, 0x04, 0x3c, 0x2d, 0x48, 0x00, 0x00, 
	0x02, 0x00, 0x84, 0x34, 0x10, 0x00, 0xa2, 0xaf, 0x2d, 0x28, 0xa0, 0x03, 0x14, 0x00, 0x06, 0x24, 
	0x2d, 0x38, 0x00, 0x00, 0x2d, 0x40, 0x00, 0x00, 0x5c, 0x02, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 
	0x8f, 0xff, 0x00, 0x10, 0x50, 0x00, 0xbf, 0xdf, 0xc0, 0x1c, 0x42, 0x8e, 0x00, 0x80, 0x04, 0x3c, 
	0x2d, 0x28, 0xa0, 0x03, 0x14, 0x00, 0x06, 0x24, 0x2d, 0x38, 0x00, 0x00, 0x2d, 0x40, 0x00, 0x00, 
	0x2d, 0x48, 0x00, 0x00, 0xf4, 0xff, 0x00, 0x10, 0x10, 0x00, 0xa2, 0xaf, 0x90, 0x04, 0x40, 0x0c, 
	0x00, 0x00, 0x00, 0x00, 0xc6, 0xff, 0x00, 0x10, 0x00, 0x01, 0x05, 0x3c, 0x00, 0x00, 0x00, 0x00, 
	0xf0, 0xff, 0xbd, 0x27, 0x00, 0x00, 0xbf, 0xff, 0x54, 0x04, 0x40, 0x0c, 0x05, 0x00, 0x04, 0x24, 
	0x00, 0x01, 0x02, 0x3c, 0x05, 0x00, 0x04, 0x24, 0x78, 0x04, 0x40, 0x0c, 0xe4, 0x1c, 0x45, 0x8c, 
	0x00, 0x00, 0xbf, 0xdf, 0x00, 0x01, 0x03, 0x3c, 0xe0, 0x1c, 0x60, 0xac, 0x08, 0x00, 0xe0, 0x03, 
	0x10, 0x00, 0xbd, 0x27, 0x00, 0x00, 0x00, 0x00, 0xff, 0x7f, 0x02, 0x3c, 0x00, 0x01, 0x03, 0x3c, 
	0xff, 0xff, 0x42, 0x34, 0xc0, 0x1c, 0x63, 0x24, 0x07, 0x00, 0x81, 0x04, 0x24, 0x10, 0x82, 0x00, 
	0x0c, 0x00, 0x63, 0x8c, 0xc0, 0x10, 0x02, 0x00, 0x21, 0x10, 0x43, 0x00, 0x04, 0x00, 0x46, 0xac, 
	0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x45, 0xac, 0xfa, 0xff, 0x00, 0x10, 0x14, 0x00, 0x63, 0x8c, 
	0x00, 0x01, 0x02, 0x3c, 0x80, 0x20, 0x04, 0x00, 0xdc, 0x1c, 0x43, 0x8c, 0x21, 0x20, 0x83, 0x00, 
	0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x82, 0x8c, 0xd0, 0xff, 0xbd, 0x27, 0x00, 0x01, 0x04, 0x3c, 
	0x20, 0x00, 0xbf, 0xff, 0x00, 0x01, 0x02, 0x3c, 0x10, 0x00, 0xb1, 0xff, 0x00, 0x00, 0xb0, 0xff, 
	0xb8, 0x1c, 0x42, 0x8c, 0xf4, 0x1c, 0x83, 0x8c, 0x03, 0x00, 0x62, 0x10, 0x00, 0x01, 0x11, 0x3c, 
	0xf4, 0x1c, 0x82, 0xac, 0xf0, 0x1c, 0x20, 0xae, 0xf0, 0x1c, 0x22, 0x8e, 0x07, 0x00, 0x40, 0x10, 
	0x2d, 0x18, 0x00, 0x00, 0x20, 0x00, 0xbf, 0xdf, 0x2d, 0x10, 0x60, 0x00, 0x10, 0x00, 0xb1, 0xdf, 
	0x00, 0x00, 0xb0, 0xdf, 0x08, 0x00, 0xe0, 0x03, 0x30, 0x00, 0xbd, 0x27, 0x0a, 0x01, 0x40, 0x0c, 
	0x2d, 0x20, 0x00, 0x00, 0x00, 0x01, 0x02, 0x3c, 0x80, 0x3c, 0x50, 0x24, 0x00, 0x80, 0x05, 0x3c, 
	0x2d, 0x20, 0x00, 0x02, 0x06, 0x00, 0xa5, 0x34, 0xe0, 0x05, 0x40, 0x0c, 0x2d, 0x30, 0x00, 0x00, 
	0xff, 0xff, 0x03, 0x3c, 0xef, 0xff, 0x40, 0x04, 0xee, 0x29, 0x63, 0x34, 0x24, 0x00, 0x02, 0x8e, 
	0x0f, 0x00, 0x03, 0x3c, 0xff, 0xff, 0x63, 0x34, 0x0f, 0x00, 0x40, 0x14, 0xfe, 0xff, 0x04, 0x24, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x63, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf5, 0xff, 0x64, 0x14, 0x00, 0x80, 0x05, 0x3c, 
	0xe8, 0xff, 0x00, 0x10, 0x2d, 0x20, 0x00, 0x02, 0x01, 0x00, 0x02, 0x24, 0x2d, 0x18, 0x00, 0x00, 
	0xd8, 0xff, 0x00, 0x10, 0xf0, 0x1c, 0x22, 0xae, 0x90, 0xfd, 0xbd, 0x27, 0x50, 0x02, 0xb4, 0xff, 
	0x2d, 0xa0, 0xe0, 0x00, 0x40, 0x02, 0xb3, 0xff, 0x2d, 0x98, 0x00, 0x01, 0x30, 0x02, 0xb2, 0xff, 
	0x2d, 0x90, 0xc0, 0x00, 0x20, 0x02, 0xb1, 0xff, 0x2d, 0x88, 0x80, 0x00, 0x10, 0x02, 0xb0, 0xff, 
	0x60, 0x02, 0xbf, 0xff, 0x3a, 0x03, 0x40, 0x0c, 0x2d, 0x80, 0xa0, 0x00, 0xff, 0xff, 0x03, 0x3c, 
	0x20, 0x00, 0x40, 0x04, 0xff, 0x29, 0x63, 0x34, 0x2d, 0x28, 0x00, 0x00, 0x10, 0x00, 0xa4, 0x27, 
	0xe4, 0x01, 0x40, 0x0c, 0x00, 0x02, 0x06, 0x24, 0x18, 0x00, 0xa4, 0x27, 0xfb, 0x00, 0x06, 0x24, 
	0xc0, 0x04, 0x40, 0x0c, 0x2d, 0x28, 0x20, 0x02, 0x03, 0x00, 0x40, 0x12, 0x13, 0x01, 0xa0, 0xa3, 
	0x1d, 0x00, 0x00, 0x56, 0xfd, 0x00, 0x03, 0x2a, 0x10, 0x00, 0xa0, 0xaf, 0x10, 0x00, 0xa7, 0x27, 
	0x00, 0x01, 0x04, 0x3c, 0x2d, 0x28, 0x60, 0x02, 0x80, 0x3c, 0x84, 0x24, 0x2d, 0x30, 0x00, 0x00, 
	0x00, 0x02, 0x08, 0x24, 0x2d, 0x48, 0xe0, 0x00, 0x08, 0x00, 0x0a, 0x24, 0x2d, 0x58, 0x00, 0x00, 
	0x2a, 0x06, 0x40, 0x0c, 0x00, 0x00, 0xa0, 0xaf, 0xff, 0xff, 0x03, 0x3c, 0x05, 0x00, 0x40, 0x04, 
	0xed, 0x29, 0x63, 0x34, 0x02, 0x00, 0x80, 0x12, 0x14, 0x00, 0xa2, 0x8f, 0x00, 0x00, 0x82, 0xae, 
	0x10, 0x00, 0xa3, 0x8f, 0x60, 0x02, 0xbf, 0xdf, 0x2d, 0x10, 0x60, 0x00, 0x50, 0x02, 0xb4, 0xdf, 
	0x40, 0x02, 0xb3, 0xdf, 0x30, 0x02, 0xb2, 0xdf, 0x20, 0x02, 0xb1, 0xdf, 0x10, 0x02, 0xb0, 0xdf, 
	0x08, 0x00, 0xe0, 0x03, 0x70, 0x02, 0xbd, 0x27, 0xfc, 0x00, 0x02, 0x24, 0x0b, 0x10, 0x03, 0x02, 
	0x14, 0x01, 0xa4, 0x27, 0x2d, 0x28, 0x40, 0x02, 0x2d, 0x30, 0x40, 0x00, 0x7c, 0x05, 0x40, 0x0c, 
	0x10, 0x00, 0xa2, 0xaf, 0xde, 0xff, 0x00, 0x10, 0x10, 0x00, 0xa7, 0x27, 0x00, 0x00, 0x00, 0x00, 
	0xa0, 0xfd, 0xbd, 0x27, 0x40, 0x02, 0xb3, 0xff, 0x2d, 0x98, 0xe0, 0x00, 0x30, 0x02, 0xb2, 0xff, 
	0x2d, 0x90, 0xc0, 0x00, 0x20, 0x02, 0xb1, 0xff, 0x2d, 0x88, 0xa0, 0x00, 0x10, 0x02, 0xb0, 0xff, 
	0x50, 0x02, 0xbf, 0xff, 0x3a, 0x03, 0x40, 0x0c, 0x2d, 0x80, 0x80, 0x00, 0xff, 0xff, 0x03, 0x3c, 
	0x21, 0x00, 0x40, 0x04, 0xff, 0x29, 0x63, 0x34, 0x2d, 0x28, 0x00, 0x02, 0x18, 0x00, 0xa4, 0x27, 
	0xc0, 0x04, 0x40, 0x0c, 0xfb, 0x00, 0x06, 0x24, 0x14, 0x01, 0xa4, 0x27, 0xfb, 0x00, 0x06, 0x24, 
	0xc0, 0x04, 0x40, 0x0c, 0x2d, 0x28, 0x20, 0x02, 0x00, 0x00, 0xa0, 0xaf, 0x10, 0x00, 0xa7, 0x27, 
	0x00, 0x01, 0x04, 0x3c, 0x2d, 0x58, 0x00, 0x00, 0x2d, 0x28, 0x60, 0x02, 0x80, 0x3c, 0x84, 0x24, 
	0x2d, 0x30, 0x00, 0x00, 0x00, 0x02, 0x08, 0x24, 0x2d, 0x48, 0xe0, 0x00, 0x10, 0x00, 0x0a, 0x24, 
	0x13, 0x01, 0xa0, 0xa3, 0x2a, 0x06, 0x40, 0x0c, 0x0f, 0x02, 0xa0, 0xa3, 0xff, 0xff, 0x03, 0x3c, 
	0x09, 0x00, 0x40, 0x04, 0xed, 0x29, 0x63, 0x34, 0x10, 0x00, 0xa4, 0x8f, 0x06, 0x00, 0x80, 0x04, 
	0x2d, 0x18, 0x80, 0x00, 0x03, 0x00, 0x40, 0x12, 0x14, 0x00, 0xa2, 0x8f, 0x00, 0x00, 0x44, 0xae, 
	0x04, 0x00, 0x42, 0xae, 0x2d, 0x18, 0x00, 0x00, 0x50, 0x02, 0xbf, 0xdf, 0x2d, 0x10, 0x60, 0x00, 
	0x40, 0x02, 0xb3, 0xdf, 0x30, 0x02, 0xb2, 0xdf, 0x20, 0x02, 0xb1, 0xdf, 0x10, 0x02, 0xb0, 0xdf, 
	0x08, 0x00, 0xe0, 0x03, 0x60, 0x02, 0xbd, 0x27, 0x30, 0xff, 0xbd, 0x27, 0x00, 0x01, 0x03, 0x3c, 
	0xc0, 0x00, 0xbf, 0xff, 0xb0, 0x00, 0xb3, 0xff, 0x70, 0x00, 0x13, 0x24, 0xa0, 0x00, 0xb2, 0xff, 
	0x18, 0x00, 0xb2, 0x27, 0x90, 0x00, 0xb1, 0xff, 0x2d, 0x88, 0x80, 0x00, 0x80, 0x00, 0xb0, 0xff, 
	0x2d, 0x80, 0xa0, 0x00, 0xb8, 0x1c, 0x62, 0x8c, 0x01, 0x00, 0x42, 0x24, 0x62, 0x01, 0x40, 0x0c, 
	0xb8, 0x1c, 0x62, 0xac, 0x84, 0x04, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x28, 0x00, 0x00, 
	0x70, 0x00, 0x06, 0x24, 0xe4, 0x01, 0x40, 0x0c, 0x2d, 0x20, 0xa0, 0x03, 0x00, 0x00, 0xb3, 0xaf, 
	0x00, 0x80, 0x02, 0x3c, 0x14, 0x00, 0xb0, 0xaf, 0x03, 0x00, 0x42, 0x34, 0x2d, 0x28, 0x20, 0x02, 
	0x4f, 0x00, 0x06, 0x24, 0x2d, 0x20, 0x40, 0x02, 0x08, 0x00, 0x20, 0x12, 0x08, 0x00, 0xa2, 0xaf, 
	0xc0, 0x04, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x20, 0x40, 0x02, 0x2e, 0x05, 0x40, 0x0c, 
	0x67, 0x00, 0xa0, 0xa3, 0x01, 0x00, 0x42, 0x24, 0x10, 0x00, 0xa2, 0xaf, 0x02, 0x00, 0x04, 0x24, 
	0x58, 0x00, 0x40, 0x0c, 0x70, 0x00, 0xbd, 0xaf, 0x78, 0x00, 0xb3, 0xaf, 0x70, 0x00, 0x05, 0x24, 
	0x2d, 0x20, 0xa0, 0x03, 0x74, 0x00, 0xa2, 0xaf, 0x44, 0x00, 0x02, 0x24, 0x94, 0x04, 0x40, 0x0c, 
	0x7c, 0x00, 0xa2, 0xaf, 0x01, 0x00, 0x05, 0x24, 0x88, 0x04, 0x40, 0x0c, 0x70, 0x00, 0xa4, 0x27, 
	0x09, 0x00, 0x40, 0x14, 0x2d, 0x18, 0x00, 0x00, 0xc0, 0x00, 0xbf, 0xdf, 0x2d, 0x10, 0x60, 0x00, 
	0xb0, 0x00, 0xb3, 0xdf, 0xa0, 0x00, 0xb2, 0xdf, 0x90, 0x00, 0xb1, 0xdf, 0x80, 0x00, 0xb0, 0xdf, 
	0x08, 0x00, 0xe0, 0x03, 0xd0, 0x00, 0xbd, 0x27, 0x04, 0x00, 0x04, 0x24, 0xe0, 0x01, 0x40, 0x0c, 
	0x01, 0x00, 0x05, 0x3c, 0x04, 0x00, 0x04, 0x24, 0xe0, 0x01, 0x40, 0x0c, 0x02, 0x00, 0x05, 0x3c, 
	0x00, 0x80, 0x04, 0x3c, 0x2d, 0x28, 0x00, 0x00, 0xe0, 0x01, 0x40, 0x0c, 0x02, 0x00, 0x84, 0x34, 
	0x00, 0x80, 0x04, 0x3c, 0xe0, 0x01, 0x40, 0x0c, 0x2d, 0x28, 0x00, 0x00, 0xea, 0xff, 0x00, 0x10, 
	0x01, 0x00, 0x03, 0x24, 0x00, 0x00, 0x00, 0x00, 0xd0, 0xff, 0xbd, 0x27, 0x10, 0x00, 0xb1, 0xff, 
	0x2d, 0x88, 0x80, 0x00, 0x20, 0x00, 0xbf, 0xff, 0x00, 0x00, 0xb0, 0xff, 0x00, 0x60, 0x10, 0x40, 
	0x01, 0x00, 0x02, 0x3c, 0x24, 0x80, 0x02, 0x02, 0x11, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x00, 
	0x7c, 0x04, 0x40, 0x0c, 0x2d, 0x20, 0x20, 0x02, 0x2d, 0x88, 0x40, 0x00, 0x0f, 0x00, 0x00, 0x00, 
	0x07, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x10, 0x20, 0x02, 0x20, 0x00, 0xbf, 0xdf, 
	0x10, 0x00, 0xb1, 0xdf, 0x00, 0x00, 0xb0, 0xdf, 0x08, 0x00, 0xe0, 0x03, 0x30, 0x00, 0xbd, 0x27, 
	0xd2, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 0xf8, 0xff, 0x00, 0x10, 0x2d, 0x10, 0x20, 0x02, 
	0xbe, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 0xed, 0xff, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 
	0xd0, 0xff, 0xbd, 0x27, 0x10, 0x00, 0xb1, 0xff, 0x2d, 0x88, 0x80, 0x00, 0x20, 0x00, 0xbf, 0xff, 
	0x00, 0x00, 0xb0, 0xff, 0x00, 0x60, 0x10, 0x40, 0x01, 0x00, 0x02, 0x3c, 0x24, 0x80, 0x02, 0x02, 
	0x11, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x00, 0x80, 0x04, 0x40, 0x0c, 0x2d, 0x20, 0x20, 0x02, 
	0x2d, 0x88, 0x40, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x00, 
	0x2d, 0x10, 0x20, 0x02, 0x20, 0x00, 0xbf, 0xdf, 0x10, 0x00, 0xb1, 0xdf, 0x00, 0x00, 0xb0, 0xdf, 
	0x08, 0x00, 0xe0, 0x03, 0x30, 0x00, 0xbd, 0x27, 0xd2, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 
	0xf8, 0xff, 0x00, 0x10, 0x2d, 0x10, 0x20, 0x02, 0xbe, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 
	0xed, 0xff, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x12, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x13, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x16, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x17, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x6b, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x77, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x89, 0xff, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x78, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0xff, 0xff, 0x19, 0x3c, 0xc0, 0xff, 0x39, 0x37, 0x26, 0x00, 0xa0, 0x18, 0x21, 0x50, 0x85, 0x00, 
	0x24, 0x40, 0x99, 0x00, 0xff, 0xff, 0x4a, 0x25, 0x24, 0x48, 0x59, 0x01, 0x23, 0x50, 0x28, 0x01, 
	0x82, 0x59, 0x0a, 0x00, 0x01, 0x00, 0x6b, 0x25, 0x07, 0x00, 0x69, 0x31, 0x08, 0x00, 0x20, 0x11, 
	0xc2, 0x50, 0x0b, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xbd, 0x0f, 0x00, 0x00, 0x00, 
	0xff, 0xff, 0x29, 0x25, 0x00, 0x00, 0x00, 0x00, 0xfa, 0xff, 0x20, 0x1d, 0x40, 0x00, 0x08, 0x25, 
	0x14, 0x00, 0x40, 0x11, 0xff, 0xff, 0x4a, 0x25, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xbd, 
	0x0f, 0x00, 0x00, 0x00, 0x40, 0x00, 0x18, 0xbd, 0x0f, 0x00, 0x00, 0x00, 0x80, 0x00, 0x18, 0xbd, 
	0x0f, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x18, 0xbd, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x01, 0x18, 0xbd, 
	0x0f, 0x00, 0x00, 0x00, 0x40, 0x01, 0x18, 0xbd, 0x0f, 0x00, 0x00, 0x00, 0x80, 0x01, 0x18, 0xbd, 
	0x0f, 0x00, 0x00, 0x00, 0xc0, 0x01, 0x18, 0xbd, 0x0f, 0x00, 0x00, 0x00, 0xed, 0xff, 0x40, 0x1d, 
	0x00, 0x02, 0x08, 0x25, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x2d, 0x40, 0x80, 0x00, 0x25, 0x38, 0xa4, 0x00, 0x10, 0x00, 0x0a, 0x24, 0x07, 0x00, 0xe2, 0x30, 
	0x08, 0x00, 0x09, 0x24, 0x54, 0x00, 0x40, 0x14, 0x0f, 0x00, 0xe2, 0x30, 0x0a, 0x48, 0x42, 0x01, 
	0x2c, 0x00, 0x40, 0x14, 0x2b, 0x10, 0xc9, 0x00, 0x4f, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 
	0x01, 0x01, 0x07, 0x3c, 0x01, 0x01, 0xe7, 0x34, 0x38, 0x3c, 0x07, 0x00, 0x01, 0x01, 0xe7, 0x34, 
	0x38, 0x3c, 0x07, 0x00, 0x01, 0x01, 0xe7, 0x34, 0x00, 0x00, 0xa3, 0x78, 0x89, 0x4b, 0xe7, 0x70, 
	0xe9, 0x1c, 0x03, 0x70, 0x80, 0x80, 0x07, 0x3c, 0x80, 0x80, 0xe7, 0x34, 0x38, 0x3c, 0x07, 0x00, 
	0x80, 0x80, 0xe7, 0x34, 0x38, 0x3c, 0x07, 0x00, 0x80, 0x80, 0xe7, 0x34, 0x48, 0x12, 0x69, 0x70, 
	0x89, 0x53, 0xe7, 0x70, 0x89, 0x14, 0x43, 0x70, 0x89, 0x14, 0x4a, 0x70, 0xa9, 0x1b, 0x44, 0x70, 
	0x25, 0x18, 0x43, 0x00, 0x37, 0x00, 0x60, 0x14, 0x2d, 0x38, 0x00, 0x01, 0x00, 0x00, 0xa3, 0x78, 
	0xf0, 0xff, 0xc6, 0x24, 0x10, 0x00, 0xa5, 0x24, 0x10, 0x00, 0xc2, 0x2c, 0x00, 0x00, 0xe3, 0x7c, 
	0x30, 0x00, 0x40, 0x14, 0x10, 0x00, 0xe7, 0x24, 0x00, 0x00, 0xa2, 0x78, 0xe9, 0x1c, 0x02, 0x70, 
	0x48, 0x12, 0x49, 0x70, 0x89, 0x14, 0x43, 0x70, 0x89, 0x14, 0x4a, 0x70, 0xa9, 0x1b, 0x44, 0x70, 
	0x25, 0x10, 0x43, 0x00, 0x1a, 0x00, 0x40, 0x50, 0x00, 0x00, 0xa3, 0x78, 0x26, 0x00, 0x00, 0x10, 
	0x2d, 0x20, 0xe0, 0x00, 0x24, 0x00, 0x40, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa3, 0xdc, 
	0x01, 0x01, 0x09, 0x3c, 0x01, 0x01, 0x29, 0x35, 0x38, 0x4c, 0x09, 0x00, 0x01, 0x01, 0x29, 0x35, 
	0x38, 0x4c, 0x09, 0x00, 0x01, 0x01, 0x29, 0x35, 0x80, 0x80, 0x0a, 0x3c, 0x80, 0x80, 0x4a, 0x35, 
	0x38, 0x54, 0x0a, 0x00, 0x80, 0x80, 0x4a, 0x35, 0x38, 0x54, 0x0a, 0x00, 0x80, 0x80, 0x4a, 0x35, 
	0x2f, 0x10, 0x69, 0x00, 0x27, 0x18, 0x03, 0x00, 0x24, 0x10, 0x43, 0x00, 0x24, 0x10, 0x4a, 0x00, 
	0x10, 0x00, 0x40, 0x14, 0x2d, 0x38, 0x00, 0x01, 0x00, 0x00, 0xa3, 0xdc, 0x00, 0x00, 0x00, 0x00, 
	0xf8, 0xff, 0xc6, 0x24, 0x08, 0x00, 0xa5, 0x24, 0x08, 0x00, 0xc2, 0x2c, 0x00, 0x00, 0xe3, 0xfc, 
	0x08, 0x00, 0x40, 0x14, 0x08, 0x00, 0xe7, 0x24, 0x00, 0x00, 0xa2, 0xdc, 0x27, 0x18, 0x02, 0x00, 
	0x2f, 0x10, 0x49, 0x00, 0x24, 0x10, 0x43, 0x00, 0x24, 0x10, 0x4a, 0x00, 0xf4, 0xff, 0x40, 0x50, 
	0x00, 0x00, 0xa3, 0xdc, 0x2d, 0x20, 0xe0, 0x00, 0x10, 0x00, 0xc0, 0x10, 0x2d, 0x10, 0xc0, 0x00, 
	0x00, 0x00, 0xa2, 0x90, 0xff, 0xff, 0xc6, 0x24, 0x01, 0x00, 0xa5, 0x24, 0x00, 0x00, 0x82, 0xa0, 
	0x00, 0x16, 0x02, 0x00, 0xf8, 0xff, 0x40, 0x14, 0x01, 0x00, 0x84, 0x24, 0x2d, 0x10, 0xc0, 0x00, 
	0x06, 0x00, 0x40, 0x10, 0xff, 0xff, 0xc6, 0x24, 0x00, 0x00, 0x80, 0xa0, 0x2d, 0x10, 0xc0, 0x00, 
	0x01, 0x00, 0x84, 0x24, 0xfc, 0xff, 0x40, 0x14, 0xff, 0xff, 0xc6, 0x24, 0x08, 0x00, 0xe0, 0x03, 
	0x2d, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x82, 0x30, 0x43, 0x00, 0x40, 0x14, 
	0x2d, 0x38, 0x80, 0x00, 0x0f, 0x00, 0x83, 0x30, 0x01, 0x01, 0x02, 0x3c, 0x01, 0x01, 0x42, 0x34, 
	0x38, 0x14, 0x02, 0x00, 0x01, 0x01, 0x42, 0x34, 0x38, 0x14, 0x02, 0x00, 0x01, 0x01, 0x42, 0x34, 
	0x1e, 0x00, 0x60, 0x14, 0x2d, 0x28, 0x80, 0x00, 0x00, 0x00, 0xa3, 0x78, 0x89, 0x43, 0x42, 0x70, 
	0x80, 0x80, 0x04, 0x3c, 0x80, 0x80, 0x84, 0x34, 0x38, 0x24, 0x04, 0x00, 0x80, 0x80, 0x84, 0x34, 
	0x38, 0x24, 0x04, 0x00, 0x80, 0x80, 0x84, 0x34, 0x48, 0x12, 0x68, 0x70, 0xe9, 0x1c, 0x03, 0x70, 
	0x89, 0x4b, 0x84, 0x70, 0x89, 0x14, 0x43, 0x70, 0x89, 0x14, 0x49, 0x70, 0xa9, 0x1b, 0x48, 0x70, 
	0x25, 0x30, 0x62, 0x00, 0x29, 0x00, 0xc0, 0x54, 0x2d, 0x20, 0xa0, 0x00, 0x10, 0x00, 0xa5, 0x24, 
	0x00, 0x00, 0xa2, 0x78, 0xe9, 0x1c, 0x02, 0x70, 0x48, 0x12, 0x48, 0x70, 0x89, 0x14, 0x43, 0x70, 
	0x89, 0x24, 0x49, 0x70, 0xa9, 0x1b, 0x86, 0x70, 0x25, 0x18, 0x64, 0x00, 0xf8, 0xff, 0x60, 0x50, 
	0x10, 0x00, 0xa5, 0x24, 0x1d, 0x00, 0x00, 0x10, 0x2d, 0x20, 0xa0, 0x00, 0x00, 0x00, 0xa3, 0xdc, 
	0x80, 0x80, 0x04, 0x3c, 0x80, 0x80, 0x84, 0x34, 0x38, 0x24, 0x04, 0x00, 0x80, 0x80, 0x84, 0x34, 
	0x38, 0x24, 0x04, 0x00, 0x80, 0x80, 0x84, 0x34, 0x2f, 0x10, 0x62, 0x00, 0x27, 0x18, 0x03, 0x00, 
	0x24, 0x10, 0x43, 0x00, 0x24, 0x10, 0x44, 0x00, 0x10, 0x00, 0x40, 0x54, 0x2d, 0x20, 0xa0, 0x00, 
	0x01, 0x01, 0x06, 0x3c, 0x01, 0x01, 0xc6, 0x34, 0x38, 0x34, 0x06, 0x00, 0x01, 0x01, 0xc6, 0x34, 
	0x38, 0x34, 0x06, 0x00, 0x01, 0x01, 0xc6, 0x34, 0x08, 0x00, 0xa5, 0x24, 0x00, 0x00, 0xa2, 0xdc, 
	0x27, 0x18, 0x02, 0x00, 0x2f, 0x10, 0x46, 0x00, 0x24, 0x10, 0x43, 0x00, 0x24, 0x10, 0x44, 0x00, 
	0xfa, 0xff, 0x40, 0x50, 0x08, 0x00, 0xa5, 0x24, 0x2d, 0x20, 0xa0, 0x00, 0x00, 0x00, 0x82, 0x80, 
	0x04, 0x00, 0x40, 0x10, 0x01, 0x00, 0x84, 0x24, 0x00, 0x00, 0x82, 0x80, 0xfb, 0xff, 0x40, 0x14, 
	0x01, 0x00, 0x84, 0x24, 0xff, 0xff, 0x84, 0x24, 0x08, 0x00, 0xe0, 0x03, 0x23, 0x10, 0x87, 0x00, 
	0x2d, 0x40, 0x80, 0x00, 0x20, 0x00, 0xc2, 0x2c, 0x1c, 0x00, 0x40, 0x14, 0x2d, 0x18, 0x00, 0x01, 
	0x25, 0x10, 0xa8, 0x00, 0x0f, 0x00, 0x42, 0x30, 0x19, 0x00, 0x40, 0x54, 0xff, 0xff, 0xc6, 0x24, 
	0x2d, 0x38, 0x00, 0x01, 0x00, 0x00, 0xa3, 0x78, 0xe0, 0xff, 0xc6, 0x24, 0x10, 0x00, 0xa5, 0x24, 
	0x20, 0x00, 0xc4, 0x2c, 0x00, 0x00, 0xe3, 0x7c, 0x10, 0x00, 0xe7, 0x24, 0x00, 0x00, 0xa2, 0x78, 
	0x10, 0x00, 0xa5, 0x24, 0x00, 0x00, 0xe2, 0x7c, 0xf6, 0xff, 0x80, 0x10, 0x10, 0x00, 0xe7, 0x24, 
	0x08, 0x00, 0xc2, 0x2c, 0x09, 0x00, 0x40, 0x14, 0x2d, 0x18, 0xe0, 0x00, 0x00, 0x00, 0xa3, 0xdc, 
	0xf8, 0xff, 0xc6, 0x24, 0x08, 0x00, 0xa5, 0x24, 0x08, 0x00, 0xc2, 0x2c, 0x00, 0x00, 0xe3, 0xfc, 
	0xfa, 0xff, 0x40, 0x10, 0x08, 0x00, 0xe7, 0x24, 0x2d, 0x18, 0xe0, 0x00, 0xff, 0xff, 0xc6, 0x24, 
	0xff, 0xff, 0x02, 0x24, 0x07, 0x00, 0xc2, 0x10, 0x2d, 0x20, 0x40, 0x00, 0x00, 0x00, 0xa2, 0x90, 
	0xff, 0xff, 0xc6, 0x24, 0x01, 0x00, 0xa5, 0x24, 0x00, 0x00, 0x62, 0xa0, 0xfb, 0xff, 0xc4, 0x14, 
	0x01, 0x00, 0x63, 0x24, 0x08, 0x00, 0xe0, 0x03, 0x2d, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 
	0x60, 0xff, 0xbd, 0x27, 0x80, 0x00, 0xb0, 0xff, 0x90, 0x00, 0xbf, 0xff, 0xd2, 0x01, 0x40, 0x0c, 
	0x00, 0x01, 0x10, 0x3c, 0xc0, 0x1c, 0x06, 0x8e, 0x2d, 0x38, 0x00, 0x00, 0x00, 0x00, 0xc2, 0x90, 
	0x22, 0x00, 0x40, 0x10, 0xc0, 0x1c, 0x10, 0x26, 0x1e, 0x00, 0x42, 0x24, 0x03, 0x29, 0x02, 0x00, 
	0x0d, 0x00, 0xa0, 0x10, 0x00, 0x00, 0xc0, 0xac, 0xff, 0xff, 0xa5, 0x24, 0xff, 0xff, 0x02, 0x24, 
	0x09, 0x00, 0xa2, 0x10, 0xff, 0xff, 0x08, 0x24, 0x00, 0x11, 0x07, 0x00, 0xff, 0xff, 0xa5, 0x24, 
	0x21, 0x20, 0x46, 0x00, 0x01, 0x00, 0xe7, 0x24, 0x00, 0x00, 0x83, 0x78, 0x21, 0x10, 0x5d, 0x00, 
	0xf9, 0xff, 0xa8, 0x14, 0x00, 0x00, 0x43, 0x7c, 0xd4, 0x06, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 
	0x08, 0x00, 0xa4, 0x8f, 0xff, 0x7f, 0x02, 0x3c, 0xff, 0xff, 0x42, 0x34, 0x24, 0x10, 0x82, 0x00, 
	0x20, 0x00, 0x43, 0x28, 0x09, 0x00, 0x60, 0x10, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x83, 0x04, 
	0x14, 0x00, 0x03, 0x8e, 0x0c, 0x00, 0x03, 0x8e, 0xc0, 0x10, 0x02, 0x00, 0x21, 0x10, 0x43, 0x00, 
	0x00, 0x00, 0x43, 0x8c, 0x09, 0x00, 0x60, 0x54, 0x04, 0x00, 0x45, 0x8c, 0x0f, 0x00, 0x00, 0x00, 
	0xd2, 0x01, 0x40, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0xb0, 0xdf, 0x90, 0x00, 0xbf, 0xdf, 
	0x2d, 0x10, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0xa0, 0x00, 0xbd, 0x27, 0x09, 0xf8, 0x60, 0x00, 
	0x2d, 0x20, 0xa0, 0x03, 0xf5, 0xff, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x90, 0xff, 0xbd, 0x27, 0x30, 0x00, 0xb1, 0xff, 0x2d, 0x88, 0x80, 0x00, 0x00, 0x01, 0x04, 0x3c, 
	0x50, 0x00, 0xb3, 0xff, 0x80, 0x1c, 0x84, 0x24, 0x40, 0x00, 0xb2, 0xff, 0x20, 0x00, 0xb0, 0xff, 
	0x2d, 0x98, 0xa0, 0x00, 0x60, 0x00, 0xbf, 0xff, 0x98, 0x06, 0x40, 0x0c, 0x2d, 0x90, 0xc0, 0x00, 
	0xff, 0xff, 0x03, 0x3c, 0x2d, 0x80, 0x40, 0x00, 0x19, 0x00, 0x40, 0x10, 0xf0, 0x29, 0x63, 0x34, 
	0x10, 0x00, 0x20, 0xae, 0x2d, 0x28, 0x40, 0x00, 0x00, 0x80, 0x04, 0x3c, 0xff, 0xff, 0x03, 0x24, 
	0x18, 0x00, 0x42, 0x8c, 0x01, 0x00, 0x4a, 0x32, 0x08, 0x00, 0x23, 0xae, 0x09, 0x00, 0x84, 0x34, 
	0x04, 0x00, 0x22, 0xae, 0x40, 0x00, 0x06, 0x24, 0x00, 0x00, 0x30, 0xae, 0x2d, 0x38, 0x00, 0x00, 
	0x2d, 0x40, 0x00, 0x00, 0x2d, 0x48, 0x00, 0x00, 0x20, 0x00, 0x13, 0xae, 0x24, 0x00, 0x20, 0xae, 
	0x14, 0x00, 0x10, 0xae, 0x0e, 0x00, 0x40, 0x11, 0x1c, 0x00, 0x11, 0xae, 0x5c, 0x02, 0x40, 0x0c, 
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x03, 0x3c, 0xef, 0x29, 0x63, 0x34, 0x0b, 0x18, 0x02, 0x00, 
	0x60, 0x00, 0xbf, 0xdf, 0x2d, 0x10, 0x60, 0x00, 0x50, 0x00, 0xb3, 0xdf, 0x40, 0x00, 0xb2, 0xdf, 
	0x30, 0x00, 0xb1, 0xdf, 0x20, 0x00, 0xb0, 0xdf, 0x08, 0x00, 0xe0, 0x03, 0x70, 0x00, 0xbd, 0x27, 
	0x01, 0x00, 0x02, 0x24, 0x2d, 0x20, 0xa0, 0x03, 0x04, 0x00, 0xa2, 0xaf, 0xc8, 0x06, 0x40, 0x0c, 
	0x08, 0x00, 0xa0, 0xaf, 0xff, 0xff, 0x03, 0x3c, 0x08, 0x00, 0x22, 0xae, 0xf0, 0xff, 0x40, 0x04, 
	0xfe, 0x29, 0x63, 0x34, 0x00, 0x80, 0x04, 0x3c, 0x2d, 0x28, 0x00, 0x02, 0x09, 0x00, 0x84, 0x34, 
	0x40, 0x00, 0x06, 0x24, 0x2d, 0x38, 0x00, 0x00, 0x2d, 0x40, 0x00, 0x00, 0x5c, 0x02, 0x40, 0x0c, 
	0x2d, 0x48, 0x00, 0x00, 0xff, 0xff, 0x03, 0x3c, 0xe5, 0xff, 0x40, 0x10, 0xef, 0x29, 0x63, 0x34, 
	0xd0, 0x06, 0x40, 0x0c, 0x08, 0x00, 0x24, 0x8e, 0xcc, 0x06, 0x40, 0x0c, 0x08, 0x00, 0x24, 0x8e, 
	0xdf, 0xff, 0x00, 0x10, 0x2d, 0x18, 0x00, 0x00, 0x40, 0xff, 0xbd, 0x27, 0x30, 0x00, 0xb1, 0xff, 
	0x2d, 0x88, 0x80, 0x00, 0x00, 0x01, 0x04, 0x3c, 0xa0, 0x00, 0xbe, 0xff, 0x80, 0x1c, 0x84, 0x24, 
	0x90, 0x00, 0xb7, 0xff, 0x80, 0x00, 0xb6, 0xff, 0x2d, 0xf0, 0x20, 0x01, 0x70, 0x00, 0xb5, 0xff, 
	0x2d, 0xb0, 0xe0, 0x00, 0x60, 0x00, 0xb4, 0xff, 0x2d, 0xa8, 0xc0, 0x00, 0x50, 0x00, 0xb3, 0xff, 
	0x2d, 0xa0, 0xa0, 0x00, 0x40, 0x00, 0xb2, 0xff, 0x2d, 0x98, 0x00, 0x01, 0x20, 0x00, 0xb0, 0xff, 
	0x2d, 0x90, 0x40, 0x01, 0xb0, 0x00, 0xbf, 0xff, 0x98, 0x06, 0x40, 0x0c, 0x2d, 0xb8, 0x60, 0x01, 
	0xff, 0xff, 0x03, 0x3c, 0x2d, 0x80, 0x40, 0x00, 0x2d, 0x00, 0x40, 0x10, 0xf0, 0x29, 0x63, 0x34, 
	0x18, 0x00, 0x42, 0x8c, 0xff, 0xff, 0x03, 0x24, 0x24, 0x00, 0x24, 0x8e, 0x02, 0x00, 0xa5, 0x32, 
	0x04, 0x00, 0x22, 0xae, 0xc0, 0x00, 0xa2, 0x8f, 0x08, 0x00, 0x23, 0xae, 0x20, 0x00, 0x22, 0xae, 
	0x01, 0x00, 0x02, 0x24, 0x00, 0x00, 0x30, 0xae, 0x20, 0x00, 0x14, 0xae, 0x30, 0x00, 0x02, 0xae, 
	0x34, 0x00, 0x04, 0xae, 0x1c, 0x00, 0x37, 0xae, 0x24, 0x00, 0x13, 0xae, 0x28, 0x00, 0x1e, 0xae, 
	0x2c, 0x00, 0x12, 0xae, 0x14, 0x00, 0x10, 0xae, 0x09, 0x00, 0xa0, 0x14, 0x1c, 0x00, 0x11, 0xae, 
	0x03, 0x00, 0x60, 0x1a, 0x2d, 0x20, 0xc0, 0x02, 0x94, 0x04, 0x40, 0x0c, 0x2d, 0x28, 0x60, 0x02, 
	0x03, 0x00, 0x40, 0x1a, 0x2d, 0x20, 0xc0, 0x03, 0x94, 0x04, 0x40, 0x0c, 0x2d, 0x28, 0x40, 0x02, 
	0x01, 0x00, 0xa2, 0x32, 0x1c, 0x00, 0x40, 0x10, 0x01, 0x00, 0x02, 0x24, 0x01, 0x00, 0xe0, 0x52, 
	0x30, 0x00, 0x00, 0xae, 0x14, 0x00, 0x28, 0x8e, 0x00, 0x80, 0x04, 0x3c, 0x2d, 0x28, 0x00, 0x02, 
	0x2d, 0x38, 0xc0, 0x02, 0x2d, 0x48, 0x60, 0x02, 0x0a, 0x00, 0x84, 0x34, 0x5c, 0x02, 0x40, 0x0c, 
	0x40, 0x00, 0x06, 0x24, 0xff, 0xff, 0x03, 0x3c, 0xef, 0x29, 0x63, 0x34, 0x0b, 0x18, 0x02, 0x00, 
	0xb0, 0x00, 0xbf, 0xdf, 0x2d, 0x10, 0x60, 0x00, 0xa0, 0x00, 0xbe, 0xdf, 0x90, 0x00, 0xb7, 0xdf, 
	0x80, 0x00, 0xb6, 0xdf, 0x70, 0x00, 0xb5, 0xdf, 0x60, 0x00, 0xb4, 0xdf, 0x50, 0x00, 0xb3, 0xdf, 
	0x40, 0x00, 0xb2, 0xdf, 0x30, 0x00, 0xb1, 0xdf, 0x20, 0x00, 0xb0, 0xdf, 0x08, 0x00, 0xe0, 0x03, 
	0xc0, 0x00, 0xbd, 0x27, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x20, 0xa0, 0x03, 0x04, 0x00, 0xa2, 0xaf, 
	0xc8, 0x06, 0x40, 0x0c, 0x08, 0x00, 0xa0, 0xaf, 0xff, 0xff, 0x03, 0x3c, 0x08, 0x00, 0x22, 0xae, 
	0xeb, 0xff, 0x40, 0x04, 0xfe, 0x29, 0x63, 0x34, 0x00, 0x80, 0x04, 0x3c, 0x14, 0x00, 0x28, 0x8e, 
	0x2d, 0x28, 0x00, 0x02, 0x2d, 0x38, 0xc0, 0x02, 0x2d, 0x48, 0x60, 0x02, 0x0a, 0x00, 0x84, 0x34, 
	0x5c, 0x02, 0x40, 0x0c, 0x40, 0x00, 0x06, 0x24, 0xff, 0xff, 0x03, 0x3c, 0xe0, 0xff, 0x40, 0x10, 
	0xef, 0x29, 0x63, 0x34, 0xd0, 0x06, 0x40, 0x0c, 0x08, 0x00, 0x24, 0x8e, 0xcc, 0x06, 0x40, 0x0c, 
	0x08, 0x00, 0x24, 0x8e, 0xda, 0xff, 0x00, 0x10, 0x2d, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0xc0, 0xff, 0xbd, 0x27, 0x20, 0x00, 0xb2, 0xff, 0x2d, 0x90, 0x80, 0x00, 0x10, 0x00, 0xb1, 0xff, 
	0x2d, 0x88, 0x00, 0x00, 0x30, 0x00, 0xbf, 0xff, 0xbe, 0x01, 0x40, 0x0c, 0x00, 0x00, 0xb0, 0xff, 
	0x08, 0x00, 0x43, 0x8e, 0x21, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x50, 0x8e, 
	0x10, 0x00, 0x02, 0x8e, 0x01, 0x00, 0x42, 0x30, 0x05, 0x00, 0x40, 0x10, 0x00, 0x00, 0x00, 0x00, 
	0x01, 0x00, 0x31, 0x26, 0x2a, 0x10, 0x23, 0x02, 0xf9, 0xff, 0x40, 0x14, 0x40, 0x00, 0x10, 0x26, 
	0x16, 0x00, 0x23, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x8e, 0x10, 0x00, 0x60, 0x10, 
	0x02, 0x00, 0x02, 0x24, 0x01, 0x00, 0x63, 0x24, 0x00, 0x00, 0x43, 0xae, 0x00, 0x14, 0x11, 0x00, 
	0x18, 0x00, 0x03, 0xae, 0x05, 0x00, 0x42, 0x34, 0x14, 0x00, 0x10, 0xae, 0xd2, 0x01, 0x40, 0x0c, 
	0x10, 0x00, 0x02, 0xae, 0x2d, 0x10, 0x00, 0x02, 0x30, 0x00, 0xbf, 0xdf, 0x20, 0x00, 0xb2, 0xdf, 
	0x10, 0x00, 0xb1, 0xdf, 0x00, 0x00, 0xb0, 0xdf, 0x08, 0x00, 0xe0, 0x03, 0x40, 0x00, 0xbd, 0x27, 
	0x01, 0x00, 0x03, 0x24, 0xf1, 0xff, 0x00, 0x10, 0x00, 0x00, 0x42, 0xae, 0xd2, 0x01, 0x40, 0x0c, 
	0x00, 0x00, 0x00, 0x00, 0xf4, 0xff, 0x00, 0x10, 0x2d, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x40, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x41, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x44, 0x00, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x88, 0xff, 0x03, 0x24, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0xe0, 0x03, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x72, 0x6f, 0x6d, 0x30, 0x3a, 0x53, 0x49, 0x4f, 0x32, 0x4d, 0x41, 0x4e, 0x00, 0x00, 0x00, 0x00, 
	0x72, 0x6f, 0x6d, 0x30, 0x3a, 0x4d, 0x43, 0x4d, 0x41, 0x4e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x72, 0x6f, 0x6d, 0x30, 0x3a, 0x4d, 0x43, 0x53, 0x45, 0x52, 0x56, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x01, 0x00, 0x00, 0x00, 0x00, 0x21, 0x00, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x29, 0x00, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x31, 0x00, 0x01, 
	0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x39, 0x00, 0x01, 0x80, 0x39, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x39, 0x00, 0x01, 
	0x20, 0x00, 0x00, 0x00, 0xc0, 0x3a, 0x00, 0x01, 0x20, 0x00, 0x00, 0x00, 0xc0, 0x3b, 0x00, 0x01, 
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

void _exec_error ( void ) {

 Exit ( 0 );

}  /* end _exec_error */

void SMS_EExec ( char* apPath ) {

 __asm__ __volatile__(
  ".set noreorder\n\t"
  "move     $s0, $a0\n\t"
  "jal      SMS_IOPReset\n\t"
  "addiu    $a0, $zero, 1\n\t"
  "lui      $a0, 0x0100\n\t"
  "la       $a1, s_Loader\n\t"
  "jal      memcpy\n\t"
  "li       $a2, %0\n\t"
  "jal      FlushCache\n\t"
  "li       $a0, 0\n\t"
  "jal      FlushCache\n\t"
  "li       $a0, 2\n\t"
  "la       $a1, _exec_error\n\t"
  ".set reorder\n\t"
  "lui      $ra, 0x0100\n\t"
  "subu     $sp, $ra, 16\n\t"
  "move     $a0, $s0\n\t"
  :: "n"(  sizeof ( s_Loader )  )
 );

}  /* end SMS_EExec */

extern int GUI_ReadButtons0 ( void );
extern int GUI_ReadButtons1 ( void );
extern int GUI_ReadButtons2 ( void );

extern unsigned char g_PadBuf1[ 256 ] __attribute__(   (  aligned( 64 ), section( ".data"  )  )   );

void SMS_EEPort2Init ( void ) {

 if ( g_Config.m_NetworkFlags & SMS_DF_GAMEPAD ) {

  PAD_OpenPort ( 1, 0, g_PadBuf1 );
  GUI_ReadButtons = GUI_ReadButtons1;

 } else if (  ( g_Config.m_NetworkFlags & SMS_DF_REMOTE ) && ( g_IOPFlags & SMS_IOPF_RMMAN )  ) {

  RC_Open ( 1 );
  GUI_ReadButtons = GUI_ReadButtons0;

 } else {

  if ( g_IOPFlags & SMS_IOPF_RMMAN2 ) {

   RCX_Open ();
   GUI_ReadButtons = GUI_ReadButtons0;

  } else GUI_ReadButtons = GUI_ReadButtons2;

 }  /* end else */

}  /* end SMS_Port2Init */
