﻿using System;
using System.Collections.Generic;
using System.IO;
using Oni.Imaging;
using Oni.Motoko;

namespace Oni.Akira
{
    internal class AkiraDatReader
    {
        #region Private data
        private InstanceDescriptor akev;
        private InstanceDescriptor agdb;
        private InstanceDescriptor pnta;
        private InstanceDescriptor plea;
        private InstanceDescriptor txca;
        private InstanceDescriptor agqg;
        private InstanceDescriptor agqc;
        private InstanceDescriptor agqr;
        private InstanceDescriptor txma;
        private InstanceDescriptor akva;
        private InstanceDescriptor akba;
        private InstanceDescriptor idxa1;
        private InstanceDescriptor idxa2;
        private InstanceDescriptor akbp;
        private InstanceDescriptor akaa;

        private PolygonMesh mesh;
        private Plane[] planes;
        private Polygon[] polygons;
        #endregion

        #region private class DatRoom

        private class DatRoom
        {
            public readonly int BspRootIndex;
            public readonly int SideListStart;
            public readonly int SideListEnd;
            public readonly int ChildIndex;
            public readonly int SiblingIndex;
            public readonly int XTiles;
            public readonly int ZTiles;
            public readonly BoundingBox BoundingBox;
            public readonly float TileSize;
            public readonly int XOrigin;
            public readonly int ZOrigin;
            public readonly RoomFlags Flags;
            public readonly Plane Floor;
            public readonly float Height;
            public readonly byte[] CompressedGridData;

            public DatRoom(InstanceDescriptor descriptor, BinaryReader reader)
            {
                BspRootIndex = reader.ReadInt32();
                reader.Skip(4);
                SideListStart = reader.ReadInt32();
                SideListEnd = reader.ReadInt32();
                ChildIndex = reader.ReadInt32();
                SiblingIndex = reader.ReadInt32();
                reader.Skip(4);
                XTiles = reader.ReadInt32();
                ZTiles = reader.ReadInt32();
                int ofsGridData = reader.ReadInt32();
                int lenGridData = reader.ReadInt32();
                TileSize = reader.ReadSingle();
                BoundingBox = reader.ReadBoundingBox();
                XOrigin = reader.ReadInt16();
                ZOrigin = reader.ReadInt16();
                reader.Skip(16);
                Flags = (RoomFlags)reader.ReadInt32();
                Floor = reader.ReadPlane();
                Height = reader.ReadSingle();

                if (ofsGridData != 0 && lenGridData != 0)
                {
                    using (BinaryReader rawReader = descriptor.GetRawReader(ofsGridData))
                        CompressedGridData = rawReader.ReadBytes(lenGridData);
                }
            }
        }

        #endregion
        #region private class DatRoomBspNode

        private class DatRoomBspNode
        {
            public readonly int PlaneIndex;
            public readonly int FrontChildIndex;
            public readonly int BackChildIndex;

            public DatRoomBspNode(BinaryReader reader)
            {
                PlaneIndex = reader.ReadInt32();
                BackChildIndex = reader.ReadInt32();
                FrontChildIndex = reader.ReadInt32();
            }
        }

        #endregion
        #region private class DatRoomSide

        private class DatRoomSide
        {
            public readonly int SideListStart;
            public readonly int SideListEnd;

            public DatRoomSide(BinaryReader reader)
            {
                reader.Skip(4);
                SideListStart = reader.ReadInt32();
                SideListEnd = reader.ReadInt32();
                reader.Skip(16);
            }
        }

        #endregion
        #region private class DatRoomAdjacency

        private class DatRoomAdjacency
        {
            public readonly int RoomIndex;
            public readonly int QuadIndex;

            public DatRoomAdjacency(BinaryReader reader)
            {
                RoomIndex = reader.ReadInt32();
                QuadIndex = reader.ReadInt32();
                reader.Skip(4);
            }
        }

        #endregion

        public static PolygonMesh Read(InstanceDescriptor akev)
        {
            var reader = new AkiraDatReader
            {
                akev = akev,
                mesh = new PolygonMesh(new MaterialLibrary())
            };

            reader.Read();
            return reader.mesh;
        }

        private void Read()
        {
            using (var reader = akev.OpenRead())
            {
                pnta = reader.ReadInstance();
                plea = reader.ReadInstance();
                txca = reader.ReadInstance();
                agqg = reader.ReadInstance();
                agqr = reader.ReadInstance();
                agqc = reader.ReadInstance();
                agdb = reader.ReadInstance();
                txma = reader.ReadInstance();
                akva = reader.ReadInstance();
                akba = reader.ReadInstance();
                idxa1 = reader.ReadInstance();
                idxa2 = reader.ReadInstance();
                akbp = reader.ReadInstance();
                reader.Skip(8);
                akaa = reader.ReadInstance();
            }

            ReadGeometry();
            ReadDebugInfo();
            ReadMaterials();
            ReadScriptIndices();
            ReadRooms();
        }

        private void ReadGeometry()
        {
            int[] planeIndices;

            using (var reader = pnta.OpenRead(52))
                mesh.Points.AddRange(reader.ReadVector3VarArray());

            using (var reader = txca.OpenRead(20))
                mesh.TexCoords.AddRange(reader.ReadVector2VarArray());

            using (var reader = plea.OpenRead(20))
                planes = reader.ReadPlaneVarArray();

            using (var reader = agqc.OpenRead(20))
            {
                planeIndices = new int[reader.ReadInt32()];

                for (int i = 0; i < planeIndices.Length; i++)
                {
                    planeIndices[i] = reader.ReadInt32();

                    //
                    // Ignore bounding boxes, we don't need them
                    //

                    reader.Skip(24);
                }
            }

            using (var reader = agqg.OpenRead(20))
            {
                polygons = new Polygon[reader.ReadInt32()];

                for (int i = 0; i < polygons.Length; i++)
                {
                    var pointIndices = reader.ReadInt32Array(4);
                    var texCoordIndices = reader.ReadInt32Array(4);
                    var colors = reader.ReadColorArray(4);
                    var flags = (GunkFlags)reader.ReadInt32();
                    int objectId = reader.ReadInt32();

                    if ((flags & GunkFlags.Triangle) != 0)
                    {
                        Array.Resize(ref pointIndices, 3);
                        Array.Resize(ref texCoordIndices, 3);
                        Array.Resize(ref colors, 3);

                        flags &= ~GunkFlags.Triangle;
                    }

                    var polygon = new Polygon(mesh, pointIndices, PlaneFromIndex(planeIndices[i]))
                    {
                        Flags = flags & ~GunkFlags.Transparent,
                        TexCoordIndices = texCoordIndices,
                        Colors = colors
                    };

                    if (objectId == -1)
                    {
                        polygon.ObjectType = -1;
                        polygon.ObjectId = -1;
                    }
                    else
                    {
                        polygon.ObjectType = (objectId >> 24) & 0xff;
                        polygon.ObjectId = objectId & 0xffffff;
                    }

                    polygons[i] = polygon;
                }
            }

            foreach (var polygon in polygons)
            {
                if ((polygon.Flags & (GunkFlags.Ghost | GunkFlags.StairsUp | GunkFlags.StairsDown)) != 0)
                    mesh.Ghosts.Add(polygon);
                else
                    mesh.Polygons.Add(polygon);
            }
        }

        private Plane PlaneFromIndex(int index)
        {
            var plane = planes[index & int.MaxValue];

            if (index < 0)
            {
                plane.Normal = -plane.Normal;
                plane.D = -plane.D;
            }

            return plane;
        }

        private void ReadMaterials()
        {
            //
            // Read material list from TXMA
            //

            Material[] materials;

            using (var reader = txma.OpenRead(20))
            {
                materials = new Material[reader.ReadInt32()];

                for (int i = 0; i < materials.Length; i++)
                {
                    var texture = reader.ReadInstance();

                    if (texture == null)
                        continue;

                    var material = mesh.Materials.GetMaterial(Utils.CleanupTextureName(texture.Name));
                    material.Image = TextureDatReader.Read(texture).Surfaces[0];

                    if (material.Image.HasAlpha)
                        material.Flags |= GunkFlags.Transparent;

                    materials[i] = material;
                }
            }

            //
            // Assign materials to polygons based on AGQR
            //

            using (var reader = agqr.OpenRead(20))
            {
                int count = reader.ReadInt32();

                for (int i = 0; i < count; i++)
                    polygons[i].Material = materials[reader.ReadInt32() & 0xffff];
            }

            //
            // Assign special materials: danger, stairs etc.
            //

            foreach (var polygon in polygons)
            {
                var marker = mesh.Materials.Markers.GetMarker(polygon);

                if (marker != null)
                    polygon.Material = marker;
            }
        }

        private void ReadScriptIndices()
        {
            if (idxa1 == null || idxa2 == null)
                return;

            int[] scriptQuadIndices;
            int[] scriptIds;

            using (var reader = idxa1.OpenRead(20))
                scriptQuadIndices = reader.ReadInt32VarArray();

            using (var reader = idxa2.OpenRead(20))
                scriptIds = reader.ReadInt32VarArray();

            for (int i = 0; i < scriptQuadIndices.Length; i++)
                polygons[scriptQuadIndices[i]].ScriptId = scriptIds[i];
        }

        private void ReadDebugInfo()
        {
            if (agdb == null)
            {
                var debugFileName = "AGDB" + akev.Name + ".oni";
                var debugFilePath = Path.Combine(Path.GetDirectoryName(akev.File.FilePath), debugFileName);

                if (!File.Exists(debugFilePath))
                    return;

                Console.WriteLine(debugFilePath);

                var debugFile = akev.File.FileManager.OpenFile(debugFilePath);

                if (debugFile == null)
                    return;

                agdb = debugFile.Descriptors[0];
            }

            if (agdb == null || agdb.Template.Tag != TemplateTag.AGDB)
                return;

            using (var reader = agdb.OpenRead(20))
            {
                int count = reader.ReadInt32();

                var fileNames = new Dictionary<int, string>();
                var objectNames = new Dictionary<int, string>();

                for (int i = 0; i < count; i++)
                {
                    int objectNameOffset = reader.ReadInt32();
                    string objectName;

                    if (!objectNames.TryGetValue(objectNameOffset, out objectName))
                    {
                        using (var rawReader = agdb.GetRawReader(objectNameOffset))
                            objectName = rawReader.ReadString(256);

                        objectName = objectName.Replace('.', '_');
                        objectNames.Add(objectNameOffset, objectName);
                    }

                    int fileNameOffset = reader.ReadInt32();
                    string fileName;

                    if (!fileNames.TryGetValue(fileNameOffset, out fileName))
                    {
                        using (var rawReader = agdb.GetRawReader(fileNameOffset))
                            fileName = rawReader.ReadString(256);

                        fileName = Path.GetFileNameWithoutExtension(fileName);
                        fileNames.Add(fileNameOffset, fileName);
                    }

                    if (!string.IsNullOrEmpty(objectName))
                        mesh.HasDebugInfo = true;

                    polygons[i].ObjectName = objectName;
                    polygons[i].FileName = fileName;
                }
            }
        }

        private void ReadRooms()
        {
            DatRoomBspNode[] bspTrees;
            DatRoomSide[] roomSides;
            DatRoomAdjacency[] roomAdjacencies;
            DatRoom[] roomsData;

            using (var reader = akbp.OpenRead(22))
            {
                bspTrees = new DatRoomBspNode[reader.ReadUInt16()];

                for (int i = 0; i < bspTrees.Length; i++)
                    bspTrees[i] = new DatRoomBspNode(reader);
            }

            using (var reader = akba.OpenRead(20))
            {
                roomSides = new DatRoomSide[reader.ReadInt32()];

                for (int i = 0; i < roomSides.Length; i++)
                    roomSides[i] = new DatRoomSide(reader);
            }

            using (var reader = akaa.OpenRead(20))
            {
                roomAdjacencies = new DatRoomAdjacency[reader.ReadInt32()];

                for (int i = 0; i < roomAdjacencies.Length; i++)
                    roomAdjacencies[i] = new DatRoomAdjacency(reader);
            }

            using (var reader = akva.OpenRead(20))
            {
                roomsData = new DatRoom[reader.ReadInt32()];

                for (int i = 0; i < roomsData.Length; i++)
                    roomsData[i] = new DatRoom(akva, reader);
            }

            var rooms = new Room[roomsData.Length];

            for (int i = 0; i < roomsData.Length; i++)
            {
                var data = roomsData[i];

                var room = new Room
                {
                    BspTree = BspNodeDataToBspNode(bspTrees, data.BspRootIndex),
                    BoundingBox = data.BoundingBox
                };

                if ((data.Flags & RoomFlags.Stairs) != 0)
                {
                    room.FloorPlane = data.Floor;
                    room.Height = data.Height;
                }
                else
                {
                    room.FloorPlane = new Plane(Vector3.Up, -data.BoundingBox.Min.Y);
                    room.Height = data.BoundingBox.Max.Y - data.BoundingBox.Min.Y;
                }

                room.Grid = RoomGrid.FromCompressedData(data.XTiles, data.ZTiles, data.CompressedGridData);
                rooms[i] = room;
            }

            for (int i = 0; i < roomsData.Length; i++)
            {
                var data = roomsData[i];
                var room = rooms[i];

                //if (data.SiblingIndex != -1)
                //    room.Sibling = rooms[data.SiblingIndex];

                //if (data.ChildIndex != -1)
                //    room.Child = rooms[data.ChildIndex];

                for (int j = data.SideListStart; j < data.SideListEnd; j++)
                {
                    var sideData = roomSides[j];

                    for (int k = sideData.SideListStart; k < sideData.SideListEnd; k++)
                    {
                        var adjData = roomAdjacencies[k];
                        var adjacentRoom = rooms[adjData.RoomIndex];
                        var ghost = polygons[adjData.QuadIndex];

                        room.Ajacencies.Add(new RoomAdjacency(adjacentRoom, ghost));
                    }
                }
            }

            mesh.Rooms.AddRange(rooms);
        }

        private RoomBspNode BspNodeDataToBspNode(DatRoomBspNode[] data, int index)
        {
            var nodeData = data[index];
            RoomBspNode front = null, back = null;

            if (nodeData.BackChildIndex != -1)
                back = BspNodeDataToBspNode(data, nodeData.BackChildIndex);

            if (nodeData.FrontChildIndex != -1)
                front = BspNodeDataToBspNode(data, nodeData.FrontChildIndex);

            return new RoomBspNode(PlaneFromIndex(nodeData.PlaneIndex), back, front);
        }
    }
}
