﻿using System;

namespace Oni
{
    internal struct Plane : IEquatable<Plane>
    {
        public Vector3 Normal;
        public float D;

        public Plane(Vector3 normal, float d)
        {
            Normal = normal;
            D = d;
        }

        public Plane(Vector3 point1, Vector3 point2, Vector3 point3)
        {
            Normal = Vector3.Normalize(Vector3.Cross(point2 - point1, point3 - point1));
            D = -Vector3.Dot(Normal, point1);
        }

        public float DotCoordinate(Vector3 point) => Vector3.Dot(Normal, point) + D;

        public float DotNormal(Vector3 value) => Vector3.Dot(Normal, value);

        public void Flip()
        {
            Normal = -Normal;
            D = -D;
        }

        public static Plane Flip(Plane plane)
        {
            plane.Normal = -plane.Normal;
            plane.D = -plane.D;

            return plane;
        }

        public static bool operator ==(Plane p1, Plane p2) => p1.D == p2.D && p1.Normal == p2.Normal;
        public static bool operator !=(Plane p1, Plane p2) => p1.D != p2.D || p1.Normal != p2.Normal;

        public bool Equals(Plane other) => other.D == D && other.Normal == Normal;

        public override bool Equals(object obj) => obj is Plane && Equals((Plane)obj);
        public override int GetHashCode() => Normal.GetHashCode() ^ D.GetHashCode();

        public override string ToString() => $"{{Normal:{Normal} D:{D}}}";

        public int Intersects(BoundingBox box)
        {
            Vector3 max, min;

            if (Normal.X >= 0.0f)
            {
                min.X = box.Min.X;
                max.X = box.Max.X;
            }
            else
            {
                min.X = box.Max.X;
                max.X = box.Min.X;
            }

            if (Normal.Y >= 0.0f)
            {
                min.Y = box.Min.Y;
                max.Y = box.Max.Y;
            }
            else
            {
                min.Y = box.Max.Y;
                max.Y = box.Min.Y;
            }

            if (Normal.Z >= 0.0f)
            {
                min.Z = box.Min.Z;
                max.Z = box.Max.Z;
            }
            else
            {
                min.Z = box.Max.Z;
                max.Z = box.Min.Z;
            }

            if (Vector3.Dot(Normal, min) + D > 0.0f)
                return 1;
            else if (Vector3.Dot(Normal, max) + D < 0.0f)
                return -1;
            else
                return 0;
        }
    }
}
