﻿using System;
using System.Collections.Generic;
using Oni;
using Oni.Metadata;

namespace Oni.Metadata
{
    internal class ObjectMetadata
    {
        internal enum TypeTag
        {
            CHAR = 0x43484152,
            CMBT = 0x434d4254,
            CONS = 0x434f4e53,
            DOOR = 0x444f4f52,
            FLAG = 0x464c4147,
            FURN = 0x4655524e,
            MELE = 0x4d454c45,
            NEUT = 0x4e455554,
            PART = 0x50415254,
            PATR = 0x50415452,
            PWRU = 0x50575255,
            SNDG = 0x534e4447,
            TRGV = 0x54524756,
            TRIG = 0x54524947,
            TURR = 0x54555252,
            WEAP = 0x57454150
        }

        [Flags]
        public enum ObjectFlags : uint
        {
            None = 0x00,
            Locked = 0x01,
            PlacedInGame = 0x02,
            Temporary = 0x04,
            Gunk = 0x08
        }

        public static readonly MetaStruct Header = new MetaStruct("Object",
            new Field(MetaType.Enum<ObjectFlags>(), "Flags"),
            new Field(MetaType.Vector3, "Position"),
            new Field(MetaType.Vector3, "Rotation")
        );

        //
        // Character Object
        //

        [Flags]
        internal enum CharacterFlags : uint
        {
            None = 0x00,
            IsPlayer = 0x01,
            RandomCostume = 0x02,
            NotInitiallyPresent = 0x04,
            NonCombatant = 0x08,
            CanSpawnMultiple = 0x10,
            Spawned = 0x20,
            Unkillable = 0x40,
            InfiniteAmmo = 0x80,
            Omniscient = 0x0100,
            HasLSI = 0x0200,
            Boss = 0x0400,
            UpgradeDifficulty = 0x0800,
            NoAutoDrop = 0x1000
        }

        internal enum CharacterTeam : uint
        {
            Konoko = 0,
            TCTF = 1,
            Syndicate = 2,
            Neutral = 3,
            SecurityGuard = 4,
            RogueKonoko = 5,
            Switzerland = 6,
            SyndicateAccessory = 7
        }

        internal enum CharacterJobType : uint
        {
            None = 0,
            Idle = 1,
            Guard = 2,
            Patrol = 3,
            TeamBatle = 4,
            Combat = 5,
            Melee = 6,
            Alarm = 7,
            Neutral = 8,
            Panic = 9
        }

        internal enum CharacterAlertStatus : uint
        {
            Lull = 0,
            Low = 1,
            Medium = 2,
            High = 3,
            Combat = 4
        }

        internal enum CharacterPursuitMode : uint
        {
            None = 0,
            Forget = 1,
            GoTo = 2,
            Wait = 3,
            Look = 4,
            Move = 5,
            Hunt = 6,
            Glanc = 7,
        }

        internal enum CharacterPursuitLostBehavior : uint
        {
            ReturnToJob = 0,
            KeepLooking = 1,
            FindAlarm = 2,
        }

        public static readonly MetaStruct Character = new MetaStruct("Character",
            new Field(MetaType.Enum<CharacterFlags>(), "Flags"),
            new Field(MetaType.String64, "Class"),
            new Field(MetaType.String32, "Name"),
            new Field(MetaType.String64, "Weapon"),
            new Field(new MetaStruct("CharacterScripts",
                new Field(MetaType.String32, "Spawn"),
                new Field(MetaType.String32, "Die"),
                new Field(MetaType.String32, "Combat"),
                new Field(MetaType.String32, "Alarm"),
                new Field(MetaType.String32, "Hurt"),
                new Field(MetaType.String32, "Defeated"),
                new Field(MetaType.String32, "OutOfAmmo"),
                new Field(MetaType.String32, "NoPath")), "Scripts"),
            new Field(MetaType.Int32, "AdditionalHealth"),
            new Field(new MetaStruct("CharacterJob",
                new Field(MetaType.Enum<CharacterJobType>(), "Type"),
                new Field(MetaType.Int16, "PatrolPathId")), "Job"),
            new Field(new MetaStruct("CharacterBehaviors",
                new Field(MetaType.Int16, "CombatId"),
                new Field(MetaType.Int16, "MeleeId"),
                new Field(MetaType.Int16, "NeutralId")), "Behaviors"),
            new Field(new MetaStruct("CharacterInventory",
                new Field(new MetaStruct("Ammo",
                    new Field(MetaType.Int16, "Use"),
                    new Field(MetaType.Int16, "Drop")), "Ammo"),
                new Field(new MetaStruct("EnergyCell",
                    new Field(MetaType.Int16, "Use"),
                    new Field(MetaType.Int16, "Drop")), "EnergyCell"),
                new Field(new MetaStruct("Hypo",
                    new Field(MetaType.Int16, "Use"),
                    new Field(MetaType.Int16, "Drop")), "Hypo"),
                new Field(new MetaStruct("Shield",
                    new Field(MetaType.Int16, "Use"),
                    new Field(MetaType.Int16, "Drop")), "Shield"),
                new Field(new MetaStruct("Invisibility",
                    new Field(MetaType.Int16, "Use"),
                    new Field(MetaType.Int16, "Drop")), "Invisibility"),
                new Field(MetaType.Padding(4))), "Inventory"),
            new Field(MetaType.Enum<CharacterTeam>(), "Team"),
            new Field(MetaType.Int32, "AmmoPercentage"),
            new Field(new MetaStruct("CharacterAlert",
                new Field(MetaType.Enum<CharacterAlertStatus>(), "Initial"),
                new Field(MetaType.Enum<CharacterAlertStatus>(), "Minimal"),
                new Field(MetaType.Enum<CharacterAlertStatus>(), "JobStart"),
                new Field(MetaType.Enum<CharacterAlertStatus>(), "Investigate")), "Alert"),
            new Field(MetaType.Int32, "AlarmGroups"),
            new Field(new MetaStruct("CharacterPursuit",
                new Field(MetaType.Enum<CharacterPursuitMode>(), "StrongUnseen"),
                new Field(MetaType.Enum<CharacterPursuitMode>(), "WeakUnseen"),
                new Field(MetaType.Enum<CharacterPursuitMode>(), "StrongSeen"),
                new Field(MetaType.Enum<CharacterPursuitMode>(), "WeakSeen"),
                new Field(MetaType.Enum<CharacterPursuitLostBehavior>(), "Lost")), "Pursuit")
        );

        //
        // Combat Behavior Object
        //

        internal enum CombatBehaviorType : uint
        {
            None = 0,
            Stare = 1,
            HoldAndFire = 2,
            FiringCharge = 3,
            Melee = 4,
            BarabasShoot = 5,
            BarabasAdvance = 6,
            BarabasMelee = 7,
            SuperNinjaFireball = 8,
            SuperNinjaAdvance = 9,
            SuperNinjaMelee = 10,
            RunForAlarm = 11,
            MutantMuroMelee = 12,
            MuroThunderbolt = 13
        }

        internal enum CombatMeleeOverride : uint
        {
            None = 0,
            IfPunched = 1,
            Cancelled = 2,
            ShortRange = 3,
            MediumRange = 4,
            AlwaysMelee = 5
        }

        internal enum CombatNoGunBehavior : uint
        {
            Melee = 0,
            Retreat = 1,
            RunForAlarm = 2
        }

        public static readonly MetaStruct CombatProfile = new MetaStruct("CombatProfile",
            new Field(MetaType.String64, "Name"),
            new Field(MetaType.Int32, "CombatId"),
            new Field(new MetaStruct("CMBTBehaviors",
                new Field(MetaType.Enum<CombatBehaviorType>(), "LongRange"),
                new Field(MetaType.Enum<CombatBehaviorType>(), "MediumRange"),
                new Field(MetaType.Enum<CombatBehaviorType>(), "ShortRange"),
                new Field(MetaType.Enum<CombatBehaviorType>(), "MediumRetreat"),
                new Field(MetaType.Enum<CombatBehaviorType>(), "LongRetreat")), "Behaviors"),
            new Field(new MetaStruct("CMBTCombat",
                new Field(MetaType.Float, "MediumRange"),
                new Field(MetaType.Enum<CombatMeleeOverride>(), "MeleeOverride"),
                new Field(MetaType.Enum<CombatNoGunBehavior>(), "NoGunBehavior"),
                new Field(MetaType.Float, "ShortRange"),
                new Field(MetaType.Float, "PursuitDistance")), "Combat"),
            new Field(new MetaStruct("CMBTPanic",
                new Field(MetaType.Int32, "Hurt"),
                new Field(MetaType.Int32, "GunFire"),
                new Field(MetaType.Int32, "Melee"),
                new Field(MetaType.Int32, "Sight")), "Panic"),
            new Field(new MetaStruct("CMBTAlarm",
                new Field(MetaType.Float, "SearchDistance"),
                new Field(MetaType.Float, "EnemyIgnoreDistance"),
                new Field(MetaType.Float, "EnemyAttackDistance"),
                new Field(MetaType.Int32, "DamageThreshold"),
                new Field(MetaType.Int32, "FightTimer")), "Alarm")
        );

        //
        // Console Object
        //

        [Flags]
        internal enum ConsoleFlags : ushort
        {
            None = 0x00,
            InitialActive = 0x08,
            Punch = 0x20,
            IsAlarm = 0x40
        }

        public static readonly MetaStruct Console = new MetaStruct("Console",
            new Field(MetaType.String63, "Class"),
            new Field(MetaType.Int16, "ConsoleId"),
            new Field(MetaType.Enum<ConsoleFlags>(), "Flags"),
            new Field(MetaType.String63, "InactiveTexture"),
            new Field(MetaType.String63, "ActiveTexture"),
            new Field(MetaType.String63, "TriggeredTexture")
        );

        //
        // Door Object
        //

        [Flags]
        internal enum DoorFlags : ushort
        {
            None = 0x00,
            InitialLocked = 0x01,
            InDoorFrame = 0x04,
            Manual = 0x10,
            DoubleDoor = 0x80,
            Mirror = 0x0100,
            OneWay = 0x0200,
            Reverse = 0x0400,
            Jammed = 0x800,
            InitialOpen = 0x1000,
        }

        public static readonly MetaStruct Door = new MetaStruct("Door",
            new Field(MetaType.String63, "Class"),
            new Field(MetaType.Int16, "DoorId"),
            new Field(MetaType.Int16, "KeyId"),
            new Field(MetaType.Enum<DoorFlags>(), "Flags"),
            new Field(MetaType.Vector3, "Center"),
            new Field(MetaType.Float, "SquaredActivationRadius"),
            new Field(MetaType.String63, "Texture1"),
            new Field(MetaType.String63, "Texture2")
        );

        //
        // Flag Object
        //

        public static readonly MetaStruct Flag = new MetaStruct("Flag",
            new Field(MetaType.Color, "Color"),
            new Field(MetaType.Int16, "Prefix"),
            new Field(MetaType.Int16, "FlagId"),
            new Field(MetaType.String128, "Notes")
        );

        //
        // Furniture Object
        //

        public static readonly MetaStruct Furniture = new MetaStruct("Furniture",
            new Field(MetaType.String32, "Class"),
            new Field(MetaType.String48, "Particle")
        );

        //
        // Melee Object
        //

        public static readonly MetaStruct MeleeProfile = new MetaStruct("MeleeProfile",
            new Field(MetaType.Int32, "MeleeId"),
            new Field(MetaType.String64, "Name"),
            new Field(MetaType.String64, "CharacterClass"),
            new Field(MetaType.Int32, "Notice"),
            new Field(new MetaStruct("MeleeDodge",
                new Field(MetaType.Int32, "Base"),
                new Field(MetaType.Int32, "Extra"),
                new Field(MetaType.Int32, "ExtraDamageThreshold")), "Dodge"),
            new Field(new MetaStruct("MeleeBlockSkill",
                new Field(MetaType.Int32, "Single"),
                new Field(MetaType.Int32, "Group")), "BlockSkill"),
            new Field(MetaType.Float, "NotBlocked"),
            new Field(MetaType.Float, "MustChangeStance"),
            new Field(MetaType.Float, "BlockedButUnblockable"),
            new Field(MetaType.Float, "BlockedButHasStagger"),
            new Field(MetaType.Float, "BlockedButHasBlockstun"),
            new Field(MetaType.Float, "Blocked"),
            new Field(MetaType.Float, "ThrowDanger"),
            new Field(MetaType.Int16, "DazedMinFrames"),
            new Field(MetaType.Int16, "DazedMaxFrames")
        );

        [Flags]
        internal enum MeleeTechniqueFlags : uint
        {
            None = 0x00,
            Interruptible = 0x01,
            GenerousDir = 0x02,
            Fearless = 0x04
        }

        public static readonly MetaStruct MeleeTechnique = new MetaStruct("MeleeTechnique",
            new Field(MetaType.String64, "Name"),
            new Field(MetaType.Enum<MeleeTechniqueFlags>(), "Flags"),
            new Field(MetaType.UInt32, "Weight"),
            new Field(MetaType.UInt32, "Importance"),
            new Field(MetaType.UInt32, "RepeatDelay")
        );

        internal enum MeleeMoveCategory
        {
            Attack = 0,
            Position = 16,
            Maneuver = 32,
            Evade = 48,
            Throw = 64
        }

        internal enum MeleeMoveAttackType
        {
            P,
            PP,
            PPP,
            PPPP,
            PF,
            PL,
            PR,
            PB,
            PD,
            PF_PF,
            PF_PF_PF,
            PL_PL,
            PL_PL_PL,
            PR_PR,
            PR_PR_PR,
            PB_PB,
            PB_PB_PB,
            PD_PD,
            PD_PD_PD,
            K,
            KK,
            KKK,
            KKKF,
            KF,
            KL,
            KR,
            KB,
            KD,
            KF_KF,
            KF_KF_KF,
            KL_KL,
            KL_KL_KL,
            KR_KR,
            KR_KR_KR,
            KB_KB,
            KB_KB_KB,
            KD_KD,
            KD_KD_KD,
            PPK,
            PKK,
            PKP,
            KPK,
            KPP,
            KKP,
            PK,
            KP,
            PPKK,
            PPKKK,
            PPKKKKK,
            HP,
            HPF,
            HK,
            HKF,
            CS_P,
            CS_K,
            C_P1,
            C_P2,
            C_PF,
            C_K1,
            C_K2,
            C_KF,
            GETUP_KF,
            GETUP_KB,
            R_P,
            R_K,
            RB_P,
            RB_K,
            RL_P,
            RL_K,
            RR_P,
            RR_K,
            R_SLIDE,
            J_P,
            J_K,
            JF_P,
            JF_PB,
            JF_K,
            JF_KB,
            JB_P,
            JB_K,
            JL_P,
            JL_K,
            JR_P,
            JR_K
        }

        internal enum MeleeMovePositionType
        {
            RunForward,
            RunLeft,
            RunRight,
            RunBack,

            JumpUp,
            JumpForward,
            JumpLeft,
            JumpRight,
            JumpBack,
            StartToCrouch,
            Crouch,
            Stand,

            CloseForward,
            CloseLeft,
            CloseRight,
            CloseBack,

            RunJumpForward,
            RunJumpLeft,
            RunJumpRight,
            RunJumpBack
        }

        internal enum MeleeMoveManeuverType
        {
            Advance,			// Duration, MinRange, ThresholdRange
            Retreat,			// Duration, MaxRange, ThresholdRange
            CircleLeft,			// Duration, MinAngle, MaxAngle
            CircleRight,		// Duration, MinAngle, MaxAngle
            Pause,				// Duration
            Crouch,				// Duration
            Jump,				// Duration
            Taunt,				// Duration
            RandomStop,			// Chance
            GetUpForward,		// Duration
            GetUpBackward,		// Duration
            GetUpRollLeft,		// Duration
            GetUpRollRight,		// Duration
            BarabasWave			// MaxRange
        }

        internal class MeleeMoveTypeInfo
        {
            public MeleeMoveManeuverType Type;
            public string[] ParamNames;

            public MeleeMoveTypeInfo(MeleeMoveManeuverType type, params string[] paramNames)
            {
                Type = type;
                ParamNames = paramNames;
            }
        }

        public static readonly MeleeMoveTypeInfo[] MeleeMoveManeuverTypeInfo = new MeleeMoveTypeInfo[]
        {
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.Advance, "Duration", "MinRange", "ThresholdRange"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.Retreat, "Duration", "MaxRange", "ThresholdRange"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.CircleLeft, "Duration", "MinAngle", "MaxAngle"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.CircleRight, "Duration", "MinAngle", "MaxAngle"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.Pause, "Duration"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.Crouch, "Duration"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.Jump, "Duration"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.Taunt, "Duration"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.RandomStop, "Chance"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.GetUpForward, "Duration"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.GetUpBackward, "Duration"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.GetUpRollLeft, "Duration"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.GetUpRollRight, "Duration"),
            new MeleeMoveTypeInfo(MeleeMoveManeuverType.BarabasWave, "MaxRange")
        };

        internal enum MeleeMoveEvadeType
        {
            JumpForward,
            JumpForward2,
            JumpBack,
            JumpBack2,
            JumpLeft,
            JumpLeft2,
            JumpRight,
            JumpRight2,
            RunJumpForward,
            RunJumpForward2,
            RunJumpBack,
            RunJumpBack2,
            RunJumpLeft,
            RunJumpLeft2,
            RunJumpRight,
            RunJumpRight2,
            RollForward,
            RollBackward,
            RollLeft,
            RollRight,
            SlideForward,
            SlideBack,
            SlideLeft,
            SlideRight
        }

        internal enum MeleeMoveThrowType
        {
            P_Front,
            K_Front,
            P_Behind,
            K_Behind,
            RP_Front,
            RK_Front,
            RP_Behind,
            RK_Behind,
            P_FrontDisarm,
            K_FrontDisarm,
            P_BehindDisarm,
            K_BehindDisarm,
            RP_FrontDisarm,
            RK_FrontDisarm,
            RP_BehindDisarm,
            RK_BehindDisarm,
            P_FrontRifDisarm,
            K_FrontRifDisarm,
            P_BehindRifDisarm,
            K_BehindRifDisarm,
            RP_FrontRifDisarm,
            RK_FrontRifDisarm,
            RP_BehindRifDisarm,
            RK_BehindRifDisarm,
            Tackle
        }

        public static readonly MetaStruct MeleeMove = new MetaStruct("MeleeMove",
            new Field(MetaType.Int32, "Type"),
            new Field(MetaType.Float, "Param1"),
            new Field(MetaType.Float, "Param2"),
            new Field(MetaType.Float, "Param3")
        );

        //
        // Neutral Object
        //

        [Flags]
        internal enum NeutralFlags : uint
        {
            None = 0x00,
            NoResume = 0x01,
            NoResumeAfterGive = 0x02,
            Uninterruptible = 0x04
        }

        [Flags]
        public enum NeutralItems : byte
        {
            None = 0,
            Shield = 1,
            Invisibility = 2,
            LSI = 4
        }

        public static readonly MetaStruct NeutralBehavior = new MetaStruct("NeutralBehavior",
            new Field(MetaType.String32, "Name"),
            new Field(MetaType.Int16, "NeutralId")
        );

        public static readonly MetaStruct NeutralBehaviorParams = new MetaStruct("NeutralBehaviorParams",
            new Field(MetaType.Enum<NeutralFlags>(), "Flags"),
            new Field(new MetaStruct("NeutralBehaviorRange",
                new Field(MetaType.Float, "Trigger"),
                new Field(MetaType.Float, "Talk"),
                new Field(MetaType.Float, "Follow"),
                new Field(MetaType.Float, "Enemy")), "Ranges"),
            new Field(new MetaStruct("NeutralehaviorSpeech",
                new Field(MetaType.String32, "Trigger"),
                new Field(MetaType.String32, "Abort"),
                new Field(MetaType.String32, "Enemy")), "Speech"),
            new Field(new MetaStruct("NeutralBehaviorScript",
                new Field(MetaType.String32, "AfterTalk")), "Script"),
            new Field(new MetaStruct("NeutralBehaviorRewards",
                new Field(MetaType.String32, "WeaponClass"),
                new Field(MetaType.Byte, "Ammo"),
                new Field(MetaType.Byte, "EnergyCell"),
                new Field(MetaType.Byte, "Hypo"),
                new Field(MetaType.Enum<NeutralItems>(), "Other")), "Rewards")
        );

        [Flags]
        public enum NeutralDialogLineFlags : ushort
        {
            None = 0x00,
            IsPlayer = 0x01,
            GiveItems = 0x02,
            AnimOnce = 0x04,
            OtherAnimOnce = 0x08
        }

        public static readonly MetaStruct NeutralBehaviorDialogLine = new MetaStruct("DialogLine",
            new Field(MetaType.Enum<NeutralDialogLineFlags>(), "Flags"),
            new Field(MetaType.Padding(2)),
            new Field(MetaType.Int16, "Anim"),
            new Field(MetaType.Int16, "OtherAnim"),
            new Field(MetaType.String32, "SpeechName")
        );

        //
        // Particle Object
        //

        [Flags]
        internal enum ParticleFlags : ushort
        {
            None = 0x00,
            NotInitiallyCreated = 0x02,
        }

        public static readonly MetaStruct Particle = new MetaStruct("Particle",
            new Field(MetaType.String64, "Class"),
            new Field(MetaType.String48, "Tag"),
            new Field(MetaType.Enum<ParticleFlags>(), "Flags"),
            new Field(MetaType.Vector2, "DecalScale")
        );

        //
        // Patrol Path Object
        //

        internal enum PatrolPathPointType
        {
            MoveToFlag = 0,
            Stop = 1,
            Pause = 2,
            LookAtFlag = 3,
            LookAtPoint = 4,
            MoveAndFaceFlag = 5,
            Loop = 6,
            MovementMode = 7,
            MoveToPoint = 8,
            LockFacing = 9,
            MoveThroughFlag = 10,
            MoveThroughPoint = 11,
            StopLooking = 12,
            FreeFacing = 13,
            GlanceAtFlagFor = 14, // unused ?
            MoveNearFlag = 15,
            LoopFrom = 16,
            Scan = 17,            // unused ?
            StopScanning = 18,
            MoveToFlagLookAndWait = 19,
            CallScript = 20,
            ForkScript = 21,
            IgnorePlayer = 22,
            FaceToFlagAndFire = 23
        }

        internal enum PatrolPathFacing
        {
            Forward = 0,
            Backward = 1,
            Left = 1,
            Right = 2,
            Stopped = 3
        }

        internal enum PatrolPathMovementMode
        {
            ByAlertLevel = 0,
            Stop = 1,
            Crouch = 2,
            Creep = 3,
            WalkNoAim = 4,
            Walk = 5,
            RunNoAim = 6,
            Run = 7
        }

        public static readonly MetaStruct PatrolPath = new MetaStruct("PatrolPath",
            new Field(MetaType.String32, "Name")
        );

        public static readonly MetaStruct PatrolPathInfo = new MetaStruct("PatrolPathInfo",
            new Field(MetaType.Int16, "PatrolId"),
            new Field(MetaType.Int16, "ReturnToNearest")
        );

        public static int GetPatrolPathPointSize(PatrolPathPointType pointType)
        {
            switch (pointType)
            {
                case PatrolPathPointType.IgnorePlayer:
                    return 1;
                case PatrolPathPointType.MoveToFlag:
                case PatrolPathPointType.LookAtFlag:
                case PatrolPathPointType.MoveAndFaceFlag:
                case PatrolPathPointType.ForkScript:
                case PatrolPathPointType.CallScript:
                    return 2;
                case PatrolPathPointType.Pause:
                case PatrolPathPointType.MovementMode:
                case PatrolPathPointType.LoopFrom:
                case PatrolPathPointType.LockFacing:
                    return 4;
                case PatrolPathPointType.MoveThroughFlag:
                case PatrolPathPointType.MoveNearFlag:
                case PatrolPathPointType.GlanceAtFlagFor:
                case PatrolPathPointType.Scan:
                    return 6;
                case PatrolPathPointType.MoveToFlagLookAndWait:
                case PatrolPathPointType.FaceToFlagAndFire:
                    return 8;
                case PatrolPathPointType.LookAtPoint:
                case PatrolPathPointType.MoveToPoint:
                    return 12;
                case PatrolPathPointType.MoveThroughPoint:
                    return 16;
                default:
                    return 0;
            }
        }

        //
        // PowerUp Object
        //

        internal enum PowerUpClass : uint
        {
            Ammo = 0x424D4D41,
            EnergyCell = 0x454D4D41,
            Hypo = 0x4F505948,
            Shield = 0x444C4853,
            Invisibility = 0x49564E49,
            LSI = 0x49534C41,
        }

        public static readonly MetaStruct PowerUp = new MetaStruct("PowerUp",
            new Field(MetaType.Enum<PowerUpClass>(), "Class")
        );

        //
        // Sound Object
        //

        internal enum SoundVolumeType
        {
            Box = 0x564C4D45,
            Sphere = 0x53504852
        }

        public static readonly MetaStruct Sound = new MetaStruct("Sound",
            new Field(MetaType.String32, "Class")
        );

        public static readonly MetaStruct SoundSphere = new MetaStruct("SoundSphere",
            new Field(MetaType.Float, "MinRadius"),
            new Field(MetaType.Float, "MaxRadius")
        );

        public static readonly MetaStruct SoundParams = new MetaStruct("SoundParams",
            new Field(MetaType.Float, "Volume"),
            new Field(MetaType.Float, "Pitch")
        );

        //
        // Trigger Volume Object
        //

        [Flags]
        public enum TriggerVolumeFlags : uint
        {
            None = 0x00,
            OneTimeEnter = 0x01,
            OneTimeInside = 0x02,
            OneTimeExit = 0x04,
            EnterDisabled = 0x08,
            InsideDisabled = 0x10,
            ExitDisabled = 0x20,
            Disabled = 0x40,
            PlayerOnly = 0x80
        }

        public static readonly MetaStruct TriggerVolume = new MetaStruct("TriggerVolume",
            new Field(MetaType.String63, "Name"),
            new Field(new MetaStruct("TriggerVolumeScripts",
                new Field(MetaType.String32, "Entry"),
                new Field(MetaType.String32, "Inside"),
                new Field(MetaType.String32, "Exit")), "Scripts"),
            new Field(MetaType.Byte, "Teams"),
            new Field(MetaType.Padding(3)),
            new Field(MetaType.Vector3, "Size"),
            new Field(MetaType.Int32, "TriggerVolumeId"),
            new Field(MetaType.Int32, "ParentId"),
            new Field(MetaType.String128, "Notes"),
            new Field(MetaType.Enum<TriggerVolumeFlags>(), "Flags")
        );

        //
        // Trigger Object
        //

        [Flags]
        public enum TriggerFlags : ushort
        {
            None = 0,
            InitialActive = 0x0008,
            ReverseAnim = 0x0010,
            PingPong = 0x0020,
        }

        public static readonly MetaStruct Trigger = new MetaStruct("Trigger",
            new Field(MetaType.String63, "Class"),
            new Field(MetaType.Int16, "TriggerId"),
            new Field(MetaType.Enum<TriggerFlags>(), "Flags"),
            new Field(MetaType.Color, "LaserColor"),
            new Field(MetaType.Float, "StartPosition"),
            new Field(MetaType.Float, "Speed"),
            new Field(MetaType.Int16, "EmitterCount"),
            new Field(MetaType.Int16, "TimeOn"),
            new Field(MetaType.Int16, "TimeOff")
        );

        //
        // Turret Object
        //

        [Flags]
        internal enum TurretTargetTeams : uint
        {
            None = 0x00,
            Konoko = 0x01,
            TCTF = 0x02,
            Syndicate = 0x04,
            Neutral = 0x08,
            SecurityGuard = 0x10,
            RogueKonoko = 0x20,
            Switzerland = 0x40,
            SyndicateAccessory = 0x80
        }

        [Flags]
        internal enum TurretFlags : ushort
        {
            None = 0,
            InitialActive = 0x0002,
        }

        public static readonly MetaStruct Turret = new MetaStruct("Turret",
            new Field(MetaType.String63, "Class"),
            new Field(MetaType.Int16, "TurretId"),
            new Field(MetaType.Enum<TurretFlags>(), "Flags"),
            new Field(MetaType.Padding(36)),
            new Field(MetaType.Enum<TurretTargetTeams>(), "TargetedTeams")
        );

        //
        // Weapon Object
        //

        public static readonly MetaStruct Weapon = new MetaStruct("Weapon",
            new Field(MetaType.String32, "Class")
        );

        internal enum EventType
        {
            None,
            Script,
            ActivateTurret,
            DeactivateTurret,
            ActivateConsole,
            DeactivateConsole,
            ActivateAlarm,
            DeactivateAlaram,
            ActivateTrigger,
            DeactivateTrigger,
            LockDoor,
            UnlockDoor
        }
    }
}
