﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace Oni
{
    internal sealed class SubtitleExporter : Exporter
    {
        public SubtitleExporter(InstanceFileManager fileManager, string outputDirPath)
            : base(fileManager, outputDirPath)
        {
        }

        protected override List<InstanceDescriptor> GetSupportedDescriptors(InstanceFile file)
        {
            return file.GetNamedDescriptors(TemplateTag.SUBT);
        }

        protected override void ExportInstance(InstanceDescriptor descriptor)
        {
            var filePath = Path.Combine(OutputDirPath, descriptor.FullName + ".txt");

            int baseOffset;
            int[] offsets;

            using (var reader = descriptor.OpenRead(16))
            {
                baseOffset = reader.ReadInt32();
                offsets = reader.ReadInt32Array(reader.ReadInt32());
            }

            using (var rawReader = descriptor.GetRawReader(baseOffset))
            using (var outStream = File.Create(filePath))
            using (var writer = new BinaryWriter(outStream))
            {
                int fileOffset = (int)rawReader.Position;
                var buffer = new List<byte>();

                foreach (int offset in offsets)
                {
                    rawReader.Position = fileOffset + offset;

                    while (true)
                    {
                        byte b = rawReader.ReadByte();

                        if (b == 0)
                        {
                            buffer.Add((byte)'=');
                            break;
                        }

                        buffer.Add(b);
                    }

                    writer.Write(buffer.ToArray());
                    buffer.Clear();

                    while (true)
                    {
                        byte b = rawReader.ReadByte();

                        if (b == 0)
                        {
                            buffer.AddRange(Encoding.UTF8.GetBytes(Environment.NewLine));
                            break;
                        }

                        buffer.Add(b);
                    }

                    writer.Write(buffer.ToArray());
                    buffer.Clear();
                }
            }
        }
    }
}
