package net.oni2.aeinstaller.backend.network;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import net.oni2.aeinstaller.backend.Settings;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Christian Illy
 */
public class DrupalJSONQuery {

	private static String getDepotUrl() {
		return Settings.getInstance().get("depot_api_url",
				"http://mods.oni2.net/?q=api/");
	}

	/**
	 * Execute an REST action through a HTTP POST query
	 * 
	 * @param resource
	 *            Resource to run on
	 * @param action
	 *            Action to call
	 * @param postData
	 *            Fieldname / value pairs to include in POST data
	 * @return JSON structure of item
	 * @throws Exception
	 *             on HTTP error
	 */
	public static JSONArray postAction(String resource, String action,
			HashMap<String, String> postData) throws Exception {
		try {
			List<NameValuePair> nvps = new ArrayList<NameValuePair>();
			for (String key : postData.keySet()) {
				nvps.add(new BasicNameValuePair(key, postData.get(key)));
			}
			HttpEntity data = new UrlEncodedFormEntity(nvps);
			return executeQuery(getDepotUrl() + resource + "/" + action
					+ ".json", data);
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * Execute an REST references query through a HTTP GET query
	 * 
	 * @param resource
	 *            Resource to run on
	 * @param index
	 *            Index of item to get the references from
	 * @param refName
	 *            Name of references type
	 * @return JSON structure of item
	 * @throws Exception
	 *             on HTTP error
	 */
	public static JSONArray getReferences(String resource, int index,
			String refName) throws Exception {
		return executeQuery(
				getDepotUrl() + resource + "/" + Integer.toString(index) + "/"
						+ refName + ".json", null);
	}

	/**
	 * Execute an REST item query through a HTTP GET query
	 * 
	 * @param resource
	 *            Resource to run on
	 * @param index
	 *            Index of item to get
	 * @param parameters
	 *            Parameters to pass (must start with ampersand "&")
	 * @return JSON structure of item
	 * @throws Exception
	 *             on HTTP error
	 */
	public static JSONArray getItem(String resource, int index,
			String parameters) throws Exception {
		return executeQuery(
				getDepotUrl() + resource + "/" + Integer.toString(index)
						+ ".json" + parameters, null);
	}

	/**
	 * Execute an REST index query through a HTTP GET query
	 * 
	 * @param resource
	 *            Resource to run on
	 * @param page
	 *            Number of page to get (for limited results, e.g. nodes), -1 to
	 *            ignore
	 * @return JSON structure of item
	 * @throws Exception
	 *             on HTTP error
	 */
	public static JSONArray getIndex(String resource, int page)
			throws Exception {
		String pageN = "";
		if (page >= 0)
			pageN = "&page=" + Integer.toString(page);
		return executeQuery(getDepotUrl() + resource + ".json" + pageN, null);
	}

	private static JSONArray executeQuery(String url, HttpEntity postData)
			throws Exception {
		BufferedReader input = null;
		HttpRequestBase httpQuery = null;

		try {
			DefaultHttpClient httpclient = new DefaultHttpClient();
			if (postData == null) {
				httpQuery = new HttpGet(url);
			} else {
				httpQuery = new HttpPost(url);
				((HttpPost) httpQuery).setEntity(postData);
			}

			HttpResponse response = httpclient.execute(httpQuery);

			int code = response.getStatusLine().getStatusCode();
			if ((code > 299) || (code < 200)) {
				throw new Exception(String.format(
						"Error fetching content (HTTP status code %d).", code));
			}

			HttpEntity entity = response.getEntity();

			input = new BufferedReader(new InputStreamReader(
					entity.getContent()));
			StringBuffer json = new StringBuffer();

			char data[] = new char[1024];
			int dataRead;
			while ((dataRead = input.read(data, 0, 1024)) != -1) {
				json.append(data, 0, dataRead);
			}

			EntityUtils.consume(entity);

			JSONArray jA = null;
			if (json.charAt(0) == '{') {
				jA = new JSONArray();
				jA.put(new JSONObject(json.toString()));
			} else
				jA = new JSONArray(json.toString());

			return jA;
		} catch (JSONException e) {
			e.printStackTrace();
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			if (httpQuery != null)
				httpQuery.releaseConnection();
			if (input != null) {
				try {
					input.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		return null;
	}
}
