package net.oni2.aeinstaller.gui;

import java.util.ArrayList;
import java.util.List;
import java.util.ResourceBundle;
import java.util.TreeMap;
import java.util.TreeSet;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JOptionPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.RowSorter;
import javax.swing.SortOrder;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableRowSorter;

import net.oni2.aeinstaller.backend.Settings;
import net.oni2.aeinstaller.backend.Settings.Platform;
import net.oni2.aeinstaller.backend.SizeFormatter;
import net.oni2.aeinstaller.backend.depot.DepotCacheUpdateProgressListener;
import net.oni2.aeinstaller.backend.depot.DepotManager;
import net.oni2.aeinstaller.backend.mods.Mod;
import net.oni2.aeinstaller.backend.mods.ModManager;
import net.oni2.aeinstaller.backend.mods.Type;
import net.oni2.aeinstaller.backend.oni.InstallProgressListener;
import net.oni2.aeinstaller.backend.oni.Installer;
import net.oni2.aeinstaller.gui.about.AboutDialog;
import net.oni2.aeinstaller.gui.modtable.DownloadSizeListener;
import net.oni2.aeinstaller.gui.modtable.ModTableFilter;
import net.oni2.aeinstaller.gui.modtable.ModTableModel;
import net.oni2.aeinstaller.gui.settings.SettingsDialog;

import org.javabuilders.BuildResult;
import org.javabuilders.annotations.DoInBackground;
import org.javabuilders.event.BackgroundEvent;
import org.javabuilders.swing.SwingJavaBuilder;
import org.simplericity.macify.eawt.ApplicationEvent;
import org.simplericity.macify.eawt.ApplicationListener;

/**
 * @author Christian Illy
 */
public class MainWin extends JFrame implements ApplicationListener,
		DownloadSizeListener {
	private static final long serialVersionUID = -4027395051382659650L;

	private ResourceBundle bundle = ResourceBundle.getBundle(getClass()
			.getName());
	@SuppressWarnings("unused")
	private BuildResult result = SwingJavaBuilder.build(this, bundle);

	private JMenu mainMenu;

	private JSplitPane contents;

	private JComboBox cmbModTypes;
	private JTable tblMods;
	private ModTableModel model;
	private TableRowSorter<ModTableModel> sorter;
	private JLabel lblDownloadSizeVal;

	private JLabel lblSubmitterVal;
	private JLabel lblCreatorVal;
	private HTMLLinkLabel lblDescriptionVal;

	private JButton btnInstall;

	/**
	 * Constructor of main window.
	 */
	public MainWin() {
		this.setTitle(SwingJavaBuilder.getConfig().getResource("appname")
				+ " - v"
				+ SwingJavaBuilder.getConfig().getResource("appversion"));

		contents.setDividerLocation(400);

		if (Settings.getPlatform() == Platform.MACOS) {
			mainMenu.setVisible(false);
		}

		getRootPane().setDefaultButton(btnInstall);
	}

	private void initModTypeBox() {
		cmbModTypes.removeAllItems();

		TreeMap<String, Type> types = new TreeMap<String, Type>();
		for (Type t : ModManager.getInstance().getTypesWithContent()) {
			types.put(t.getName(), t);
		}
		for (Type t : types.values()) {
			cmbModTypes.addItem(t);
		}
		cmbModTypes.setSelectedIndex(0);
	}

	private void initTable() {
		tblMods.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		tblMods.getSelectionModel().addListSelectionListener(
				new ListSelectionListener() {

					@Override
					public void valueChanged(ListSelectionEvent e) {
						int viewRow = tblMods.getSelectedRow();
						if (viewRow < 0) {
							modSelection(null);
						} else {
							int modelRow = tblMods
									.convertRowIndexToModel(viewRow);
							Mod mod = (Mod) model.getValueAt(modelRow, -1);
							modSelection(mod);
						}
					}
				});

		// To get checkbox-cells with background of row
		((JComponent) tblMods.getDefaultRenderer(Boolean.class))
				.setOpaque(true);

		model = new ModTableModel();
		model.addDownloadSizeListener(this);

		tblMods.setModel(model);

		sorter = new TableRowSorter<ModTableModel>(model);
		tblMods.setRowSorter(sorter);

		sorter.setRowFilter(new ModTableFilter(null));

		sorter.setSortable(2, false);

		List<RowSorter.SortKey> sortKeys = new ArrayList<RowSorter.SortKey>();
		sortKeys.add(new RowSorter.SortKey(0, SortOrder.ASCENDING));
		sorter.setSortKeys(sortKeys);

		for (int i = 0; i < model.getColumnCount(); i++) {
			model.setColumnConstraints(i, tblMods.getColumnModel().getColumn(i));
		}

		// for (int i = 3; i > 0; i--) {
		// tblMods.getColumnModel().removeColumn(tblMods.getColumnModel().getColumn(i));
		// }
	}

	private boolean askClose() {
		int res = JOptionPane.showConfirmDialog(this,
				bundle.getString("askClose.text"),
				bundle.getString("askClose.title"), JOptionPane.YES_NO_OPTION,
				JOptionPane.QUESTION_MESSAGE);
		return res == JOptionPane.YES_OPTION;
	}

	private void exit() {
		setVisible(false);
		dispose();
	}

	private void saveLocalData() {
		Settings.getInstance().serializeToFile();
		DepotManager.getInstance().saveToFile(Settings.getDepotCacheFilename());
	}

	@DoInBackground(progressMessage = "updateDepot.title", cancelable = false, indeterminateProgress = false)
	private void execDepotUpdate(final BackgroundEvent evt) {
		try {
			DepotManager.getInstance().updateInformation(false,
					new DepotCacheUpdateProgressListener() {

						@Override
						public void cacheUpdateProgress(String stepName,
								int current, int total) {
							evt.setProgressEnd(total);
							evt.setProgressValue(current);
							evt.setProgressMessage(stepName);
						}
					});
			ModManager.getInstance().init();
			initTable();
			initModTypeBox();

			tblMods.setVisible(true);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	@SuppressWarnings("unused")
	private void checkUpdates() {
		if (Settings.getInstance().get("notifyupdates", true)) {
		}
	}

	@SuppressWarnings("unused")
	private void focus() {
		SwingUtilities.invokeLater(new Runnable() {

			@Override
			public void run() {
				toFront();
				repaint();
			}
		});

	}

	private void showSettings() {
		new SettingsDialog().setVisible(true);
	}

	private void showAbout() {
		new AboutDialog().setVisible(true);
	}

	@SuppressWarnings("unused")
	private void loadConfig() {
		// TODO method stub
		JOptionPane.showMessageDialog(this, "loadConfig", "todo",
				JOptionPane.INFORMATION_MESSAGE);
	}

	@SuppressWarnings("unused")
	private void saveConfig() {
		// TODO method stub
		JOptionPane.showMessageDialog(this, "saveConfig", "todo",
				JOptionPane.INFORMATION_MESSAGE);
	}

	@DoInBackground(progressMessage = "initializingEdition.title", cancelable = false, indeterminateProgress = false)
	private void reglobalize(final BackgroundEvent evt) {
		Installer.initializeEdition(new InstallProgressListener() {
			@Override
			public void installProgressUpdate(int done, int total, String step) {
				evt.setProgressEnd(total);
				evt.setProgressValue(done);
				evt.setProgressMessage(step);
			}
		});
	}

	@SuppressWarnings("unused")
	private void tools() {
		// TODO method stub
		JOptionPane.showMessageDialog(this, "tools", "todo",
				JOptionPane.INFORMATION_MESSAGE);
	}

	@SuppressWarnings("unused")
	private void revertSelection() {
		// TODO method stub
		JOptionPane.showMessageDialog(this, "revertSelection", "todo",
				JOptionPane.INFORMATION_MESSAGE);
	}

	@DoInBackground(progressMessage = "mandatoryFiles.title", cancelable = false, indeterminateProgress = false)
	private void checkMandatoryFiles(final BackgroundEvent evt) {
		//TODO
		System.out.println("Mandatory Tools:");
		for (Mod m : ModManager.getInstance().getMandatoryTools()) {
			System.out.format("  %05d %15s - Local: %b - Update: %b", m.getPackageNumber(), m.getName(), m.isLocalAvailable(), m.isNewerAvailable());
		}
		System.out.println();
		
		System.out.println("Mandatory Mods:");
		for (Mod m : ModManager.getInstance().getMandatoryMods()) {
			System.out.format("  %05d %15s - Local: %b - Update: %b", m.getPackageNumber(), m.getName(), m.isLocalAvailable(), m.isNewerAvailable());
		}
	}

	@DoInBackground(progressMessage = "installing.title", cancelable = false, indeterminateProgress = false)
	private void install(final BackgroundEvent evt) {
		// TODO: Conflicts/Dependencies

		TreeSet<Mod> mods = new TreeSet<Mod>();
		mods.addAll(ModManager.getInstance().getMandatoryMods());
		mods.addAll(model.getSelectedMods());

		System.out.println("Install mods:");
		for (Mod m : mods) {
			System.out
					.println("  " + m.getPackageNumber() + ": " + m.getName());
		}

		Installer.install(mods, new InstallProgressListener() {
			@Override
			public void installProgressUpdate(int done, int total, String step) {
				evt.setProgressEnd(total);
				evt.setProgressValue(done);
				evt.setProgressMessage(step);
			}
		});
	}

	private void modSelection(Mod m) {
		lblSubmitterVal.setText("");
		lblCreatorVal.setText("");
		lblDescriptionVal.setText("");
		if (m != null) {
			lblSubmitterVal.setText(m.getName());
			lblCreatorVal.setText(m.getCreator());
			lblDescriptionVal.setText(m.getDescription());
		}
		// TODO
	}

	@SuppressWarnings("unused")
	private void modTypeSelection() {
		Type t = (Type) cmbModTypes.getSelectedItem();
		if (t != null)
			sorter.setRowFilter(new ModTableFilter(t));
		else
			sorter.setRowFilter(new ModTableFilter(null));
	}

	@Override
	public void downloadSizeChanged(int newSize) {
		lblDownloadSizeVal.setText(SizeFormatter.format(newSize, 2));
	}

	@DoInBackground(progressMessage = "initializingEdition.title", cancelable = false, indeterminateProgress = false)
	private void initialize(final BackgroundEvent evt) {
		if (!Installer.isEditionInitialized()) {
			int res = JOptionPane.showConfirmDialog(this,
					bundle.getString("askInitialize.text"),
					bundle.getString("askInitialize.title"),
					JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
			if (res == JOptionPane.NO_OPTION) {
				exit();
			} else {
				Installer.initializeEdition(new InstallProgressListener() {
					@Override
					public void installProgressUpdate(int done, int total,
							String step) {
						evt.setProgressEnd(total);
						evt.setProgressValue(done);
						evt.setProgressMessage(step);
					}
				});
			}
		}
	}

	@Override
	public void handleAbout(ApplicationEvent event) {
		event.setHandled(true);
		showAbout();
	}

	@Override
	public void handleOpenApplication(ApplicationEvent event) {
	}

	@Override
	public void handleOpenFile(ApplicationEvent event) {
	}

	@Override
	public void handlePreferences(ApplicationEvent event) {
		showSettings();
	}

	@Override
	public void handlePrintFile(ApplicationEvent event) {
	}

	@Override
	public void handleQuit(ApplicationEvent event) {
		if (askClose()) {
			event.setHandled(true);
			saveLocalData();
			exit();
		} else {
			event.setHandled(false);
		}
	}

	@Override
	public void handleReOpenApplication(ApplicationEvent event) {
	}

}