unit Functions;

interface

uses TypeDefs, Classes;

function BoolToStr(bool: Boolean): String;
function HexToLong(hex: String): LongWord;
function Decode_Int(buffer: TByteData): LongWord;
function Encode_Int(input: LongWord): TByteData;
function Decode_Float(buffer: TByteData): Single;
function Encode_Float(input: Single): TByteData;
function IntToBin(Value: Byte): String;
function DataToBin(Data: TByteData): String;
function BinToInt(bin: String): Byte;

function StringSmaller(string1, string2: String): Boolean;

function FormatNumber(Value: LongWord; Width: Byte; leadingzeros: Char): String;
function FormatFileSize(size: LongWord): String;
function CreateHexString(Data: TByteData; HexOnly: Boolean): String;
function DecodeHexString(hex: String): TByteData;
function GetWinFileName(Name: String): String;
//function GetExtractPath: String;

function Explode(_string: String; delimiter: Char): TStrings;


implementation

uses SysUtils, Math, StrUtils;

type
  TValueSwitcher = record
    case IsFloat: Boolean of
      True: (ValueFloat: Single);
      False: (ValueInt: LongWord);
  end;



function BoolToStr(bool: Boolean): String;
begin
  if bool then
    Result := 'true'
  else
    Result := 'false';
end;


function HexToLong(hex: String): LongWord;

  function NormalizeHexString(var hex: String): Boolean;
  var
    i: Byte;
  begin
    Result := True;
    if hex[1] = '$' then
    begin
      for i := 1 to Length(hex) - 1 do
        hex[i] := hex[i + 1];
      SetLength(hex, Length(hex) - 1);
    end;
    if (hex[1] = '0') and (UpCase(hex[2]) = 'X') then
    begin
      for i := 1 to Length(hex) - 2 do
        hex[i] := hex[i + 2];
      SetLength(hex, Length(hex) - 2);
    end;
    if Length(hex) = 0 then
      Result := False;
  end;

begin
  if NormalizeHexString(hex) then
    Result := StrToInt(hex)
  else
    Result := 0;
end;


function Decode_Int(buffer: TByteData): LongWord;
begin
  Result := buffer[0] + buffer[1] * 256 + buffer[2] * 256 * 256 + buffer[3] * 256 * 256 * 256;
end;


function Encode_Int(input: LongWord): TByteData;
begin
  SetLength(Result, 4);
  Result[0] := input mod 256;
  input     := input div 256;
  Result[1] := input mod 256;
  input     := input div 256;
  Result[2] := input mod 256;
  input     := input div 256;
  Result[3] := input mod 256;
end;


function Decode_Float(buffer: TByteData): Single;
var
  _valueswitcher: TValueSwitcher;
begin
  _valueswitcher.ValueInt := Decode_Int(buffer);
  Result := _valueswitcher.ValueFloat;
  if IsNAN(Result) then
    Result := 0.0;
end;


function Encode_Float(input: Single): TByteData;
var
  _valueswitcher: TValueSwitcher;
begin
  _valueswitcher.ValueFloat := input;
  Result := Encode_Int(_valueswitcher.ValueInt);
end;


function IntToBin(Value: Byte): String;
var
  i: Byte;
begin
  Result := '';
  for i := 7 downto 0 do
    Result := Result + IntToStr((Value shr i) and $01);
end;


function DataToBin(Data: TByteData): String;
var
  i, j:     Byte;
  singlebyte: Byte;
  bytepart: String;
begin
  SetLength(bytepart, 8);
  Result := '';
  for i := 0 to High(Data) do
  begin
    singlebyte := Data[i];
    for j := 7 downto 0 do
    begin
      bytepart[j + 1] := Char((singlebyte and $01) + 48);
      singlebyte      := singlebyte shr 1;
    end;
    Result := Result + bytepart + ' ';
  end;
end;


function BinToInt(bin: String): Byte;
var
  Add: Integer;
  i:   Byte;
begin
  Result := 0;
  if Length(bin) <> 8 then
    Exit;
  Add := 1;
  for i := 8 downto 1 do
  begin
    if not (bin[i] in ['0', '1']) then
      Exit;
    if bin[i] = '1' then
      Inc(Result, Add);
    Add := Add shl 1;
  end;
end;



function FormatNumber(Value: LongWord; Width: Byte; leadingzeros: Char): String;
begin
  Result := AnsiReplaceStr(Format('%' + IntToStr(Width) + 'u', [Value]), ' ', leadingzeros);
end;




function FormatFileSize(size: LongWord): String;
var
  floatformat: TFormatSettings;
begin
  floatformat.DecimalSeparator := '.';
  if size >= 1000 * 1024 * 1024 then
    Result := FloatToStrF(size / 1024 / 1024 / 1024, ffFixed, 5, 1, floatformat) + ' GB'
  else
    if size >= 1000 * 1024 then
      Result := FloatToStrF(size / 1024 / 1024, ffFixed, 5, 1, floatformat) + ' MB'
    else
      if size >= 1000 then
        Result := FloatToStrF(size / 1024, ffFixed, 5, 1, floatformat) + ' KB'
      else
        Result := IntToStr(size) + ' B';
end;




function CreateHexString(Data: TByteData; HexOnly: Boolean): String;
var
  string_build, ascii_version: String;
  i: LongWord;
begin
  string_build  := '';
  ascii_version := '';
  for i := 0 to High(Data) do
  begin
    if not HexOnly then
      if (i mod 16) = 0 then
        string_build := string_build + '0x' + IntToHex(i, 6) + '  ';
    string_build := string_build + IntToHex(Data[i], 2);
    if not HexOnly then
    begin
      if Data[i] >= 32 then
        ascii_version := ascii_version + Chr(Data[i])
      else
        ascii_version := ascii_version + '.';
      if ((i + 1) mod 2) = 0 then
        string_build := string_build + #32;
      if ((i + 1) mod 16) = 0 then
      begin
        string_build  := string_build + #32 + ascii_version + CrLf;
        ascii_version := '';
      end;
    end;
  end;
  Result := string_build;
end;




function DecodeHexString(hex: String): TByteData;
var
  i: LongWord;
begin
  SetLength(Result, Length(hex) div 2);
  for i := 0 to Length(Result) do
  begin
    Result[i] := 0;
    case UpCase(hex[1 + i * 2]) of
      '0'..'9':
        Result[i] := (Ord(UpCase(hex[1 + i * 2])) - 48) * 16;
      'A'..'F':
        Result[i] := (Ord(UpCase(hex[1 + i * 2])) - 55) * 16;
    end;
    case UpCase(hex[1 + i * 2 + 1]) of
      '0'..'9':
        Result[i] := Result[i] + (Ord(UpCase(hex[1 + i * 2 + 1])) - 48);
      'A'..'F':
        Result[i] := Result[i] + (Ord(UpCase(hex[1 + i * 2 + 1])) - 55);
    end;
  end;
end;




function StringSmaller(string1, string2: String): Boolean;
var
  i:   Integer;
  len: Integer;
begin
  len := Min(Length(string1), Length(string2));
  for i := 1 to len do
    if Ord(string1[i]) <> Ord(string2[i]) then
    begin
      Result := Ord(string1[i]) < Ord(string2[i]);
      Exit;
    end;
  Result := Length(string1) < Length(string2);
end;



function Explode(_string: String; delimiter: Char): TStrings;
var
  start, len: Word;
begin
  Result := TStringList.Create;
  start := 1;
  while PosEx(delimiter, _string, start) > 0 do
  begin
    len := PosEx(delimiter, _string, start) - start;
    Result.Add(MidStr(_string, start, len));
    start := start + len + 1;
  end;
  Result.Add(MidStr(_string, start, Length(_string) - start + 1));
end;


function GetWinFileName(Name: String): String;
begin
  Result := Name;
  Result := AnsiReplaceStr(Result, '\', '__');
  Result := AnsiReplaceStr(Result, '/', '__');
  Result := AnsiReplaceStr(Result, '>', '__');
  Result := AnsiReplaceStr(Result, '<', '__');
end;


end.