/*
 *
Copyright (C) 2017  Fábio Bento (random-guy)

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*/

#include "projectfilevago.h"

const QString ProjectFileVago::XMLTableName = "XML";
const QString ProjectFileVago::TexturesTableName = "Textures";
const QString ProjectFileVago::CharactersTableName = "Characters";
const QString ProjectFileVago::ObjectsTableName = "Objects";
const QString ProjectFileVago::LevelsTableName = "Levels";
const QString ProjectFileVago::MiscTableName = "Misc";

ProjectFileVago::ProjectData ProjectFileVago::readProjectDataFromFile(const QString &fileFullPath){

    ProjectFileVago::ProjectData currentProjectData;

    upgradeProjectFileIfNecessary(fileFullPath);

    pugi::xml_document doc;

    pugi::xml_parse_result result = doc.load_file(QSTR_TO_CSTR(fileFullPath));

    if(result.status!=pugi::status_ok){
        throw std::runtime_error(QSTR_TO_CSTR(QString("An error ocurred while loading project file.\n") + result.description()));
    }

    if(QString(doc.root().first_child().name()) != "VagoProject"){
        throw std::runtime_error(QSTR_TO_CSTR(QString(doc.root().name()) + "The file opened is not a valid Vago project file. Load aborted."));
    }

    QString projVagoVersion;

    try{
        projVagoVersion = QString(doc.select_node("/VagoProject/@vagoVersion").attribute().value());
    }
    catch (const pugi::xpath_exception& e)
    {
        throw std::runtime_error(QSTR_TO_CSTR(QString("Couldn't find the vagoVersion of the current project. Load aborted.\n") + e.what()));
    }

    if(!projVagoVersion.startsWith(GlobalVars::LastCompatibleVersion)){
        throw std::runtime_error("The project that you are trying to load seems it is not compatible with your Vago Version. Please update Vago and try again.");
    }

    // After the initial validations begin loading the project data

    auto fFetchFromToForTab = [](pugi::xml_document &doc, const QString &tableName, ProjectTable &tableToInjectData) -> void{
        tableToInjectData.from = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+tableName+"/@from")).attribute().value();
        tableToInjectData.to = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+tableName+"/@to")).attribute().value();
    };

    auto fFetchTabTableRows = [](pugi::xml_document &doc, const QString &tableName) -> QVector<ProjectTableRow>{

        QVector<ProjectTableRow> rows;

        for(const pugi::xpath_node &xPathNode : doc.select_nodes(QSTR_TO_CSTR("/VagoProject/"+tableName+"/Row"))){

            ProjectTableRow currentRow;

            pugi::xml_node currNode = xPathNode.node();

            currentRow.fileFolder = currNode.attribute("fileFolder").value();
            currentRow.fromTo = currNode.attribute("fromTo").value();
            currentRow.command = currNode.attribute("command").value();

            pugi::xml_attribute disabledAttr = currNode.attribute("disabled");
            currentRow.isDisabled = disabledAttr.empty() ? false : disabledAttr.as_bool();

            rows.append(currentRow);
        }

        return rows;
    };

    QString currentTableName = XMLTableName;

    // XML tab

    fFetchFromToForTab(doc, currentTableName, currentProjectData.xmlTable);
    currentProjectData.xmlTable.rows = fFetchTabTableRows(doc, currentTableName);

    // Textures tab

    currentTableName = TexturesTableName;

    fFetchFromToForTab(doc, currentTableName, currentProjectData.texturesTable);

    currentProjectData.texturesTable.rbTexturesType = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@type")).attribute().value();
    currentProjectData.texturesTable.cbGenMipMaps = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@genMipMaps")).attribute().as_bool();
    currentProjectData.texturesTable.cbNoUwrap = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@noUwrap")).attribute().as_bool();
    currentProjectData.texturesTable.cbNoUwrap = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@noVwrap")).attribute().as_bool();
    currentProjectData.texturesTable.cbLarge = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@large")).attribute().as_bool();
    currentProjectData.texturesTable.cbEnvMap = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@envMap")).attribute().as_bool();
    currentProjectData.texturesTable.leEnvMapTexture = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@envMapValue")).attribute().value();

    currentProjectData.texturesTable.rows = fFetchTabTableRows(doc, currentTableName);

    // Characters tab

    currentTableName = CharactersTableName;

    fFetchFromToForTab(doc, currentTableName, currentProjectData.charactersTable);

    currentProjectData.charactersTable.cbCellShading = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@cellShading")).attribute().as_bool();
    currentProjectData.charactersTable.cbNormals = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@normals")).attribute().as_bool();
    currentProjectData.charactersTable.cbStandingPose = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@standingPose")).attribute().as_bool();
    currentProjectData.charactersTable.cbWithTRBS_ONCC = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@extractTRBSONCC")).attribute().as_bool();
    currentProjectData.charactersTable.leTRBS_ONCC = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@extractTRBSONCCValue")).attribute().value();

    currentProjectData.charactersTable.rows = fFetchTabTableRows(doc, currentTableName);

    // Objects tab

    currentTableName = ObjectsTableName;

    fFetchFromToForTab(doc, currentTableName, currentProjectData.objectsTable);

    currentProjectData.objectsTable.cbTexture = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@texture")).attribute().as_bool();
    currentProjectData.objectsTable.leTextureName = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@textureValue")).attribute().value();
    currentProjectData.objectsTable.cbWithAnimation = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@withAnimation")).attribute().as_bool();
    currentProjectData.objectsTable.leAnimationName = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@withAnimationValue")).attribute().value();

    currentProjectData.objectsTable.rows = fFetchTabTableRows(doc, currentTableName);

    // Levels tab

    currentTableName = LevelsTableName;

    fFetchFromToForTab(doc, currentTableName, currentProjectData.levelsTable);

    currentProjectData.levelsTable.cbSpecificFilesLevels = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@extractWithFiles")).attribute().as_bool();
    currentProjectData.levelsTable.leSpecificFilesLevels = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@extractWithFilesValue")).attribute().value();
    currentProjectData.levelsTable.cbDatLevels = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@datFilename")).attribute().as_bool();
    currentProjectData.levelsTable.leTargetDatLevels = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@datFilenameValue")).attribute().value();
    currentProjectData.levelsTable.cbBnvLevels = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@bnvSource")).attribute().as_bool();
    currentProjectData.levelsTable.leBnvLevels = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@bnvSourceValue")).attribute().value();
    currentProjectData.levelsTable.cbGridsLevels = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@generateGrids")).attribute().as_bool();
    currentProjectData.levelsTable.cbAdditionalSourcesLevels = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@additionalSources")).attribute().as_bool();
    currentProjectData.levelsTable.leAdditSourcesLevels = doc.select_node(QSTR_TO_CSTR("/VagoProject/"+currentTableName+"/Options/@additionalSourcesValue")).attribute().value();

    currentProjectData.levelsTable.rows = fFetchTabTableRows(doc, currentTableName);

    // Misc tab

    currentTableName = MiscTableName;

    fFetchFromToForTab(doc, currentTableName, currentProjectData.miscTable);

    currentProjectData.miscTable.rows = fFetchTabTableRows(doc, currentTableName);

    return currentProjectData;
}

void ProjectFileVago::upgradeProjectFileIfNecessary(const QString &filePath){

    pugi::xml_document doc;

    pugi::xml_parse_result result = doc.load_file(QSTR_TO_CSTR(filePath));

    if(result.status!=pugi::status_ok){
        throw std::runtime_error(QSTR_TO_CSTR(QString("An error ocurred while loading project file.\n") + result.description()));
    }

    QString projectVersion = QString(doc.select_single_node("/VagoProject").node().attribute("vagoVersion").as_string());

    // 1.4 added standing pose in characters tab (we need to add it in 1.0 projects)
    if(projectVersion == "1.0"){

        if(!Util::FileSystem::backupFile(filePath, filePath + UtilVago::getDateTimeFormatForFilename(QDateTime::currentDateTime()))){
            QString errorMessage = "Couldn't backup the existing project file for version upgrade, program can't proceed.";
            Util::Dialogs::showError(errorMessage);
            LOG_FATAL << errorMessage;
            exit(1);
        }

        // Update version
        doc.select_single_node("/VagoProject").node().attribute("vagoVersion").set_value(QSTR_TO_CSTR(GlobalVars::LastCompatibleVersion));

        // Add standing pose to characters options
        doc.select_node("/VagoProject/Characters/Options").node().append_attribute("standingPose").set_value("false");

        if(!doc.save_file(QSTR_TO_CSTR(filePath), PUGIXML_TEXT("\t"), pugi::format_default | pugi::format_write_bom, pugi::xml_encoding::encoding_utf8)){
            throw std::runtime_error(QSTR_TO_CSTR("Error while saving: '" + filePath + "'. After file version upgrade."));
        }
    }
    else if(projectVersion != GlobalVars::LastCompatibleVersion){
        throw std::runtime_error("Can't load the project file, it is from an incompatible version. Probably newer?");
    }
}

// Right now it always replaces totally the old file with new content (old content is not modified but replaced)
void ProjectFileVago::saveProjectDataToFile(const QString &fileFullPath, const ProjectFileVago::ProjectData &newProjectData){
    pugi::xml_document doc;
    pugi::xml_node rootNode;

    rootNode = doc.append_child("VagoProject"); // create

    rootNode.append_attribute("vagoVersion").set_value(QSTR_TO_CSTR(GlobalVars::LastCompatibleVersion));

    // Let's starting writting the GUI data

    QString currentTableName;

    // Returns node of the table
    auto fWriteTabGenericData = [](pugi::xml_node &rootNode, const ProjectTable &table, const QString &tableName) -> pugi::xml_node{

        pugi::xml_node rowsNode = rootNode.append_child(QSTR_TO_CSTR(tableName));

        rowsNode.append_attribute("from").set_value(QSTR_TO_CSTR(table.from));
        rowsNode.append_attribute("to").set_value(QSTR_TO_CSTR(table.to));

        for(const ProjectTableRow &currRow : table.rows){

            pugi::xml_node currentRow = rowsNode.append_child("Row");

            currentRow.append_attribute("fileFolder").set_value(QSTR_TO_CSTR(currRow.fileFolder));
            currentRow.append_attribute("fromTo").set_value(QSTR_TO_CSTR(currRow.fromTo));
            currentRow.append_attribute("command").set_value(QSTR_TO_CSTR(currRow.command));

            // Only necessary to add if it is infact disabled
            if(currRow.isDisabled){
                currentRow.append_attribute("disabled").set_value(currRow.isDisabled);
            }
        }

        return rowsNode;
    };

    // XML tab

    currentTableName = XMLTableName;

    fWriteTabGenericData(rootNode, newProjectData.xmlTable, currentTableName);

    // Textures tab

    currentTableName = TexturesTableName;

    pugi::xml_node currentNodeTable = fWriteTabGenericData(rootNode, newProjectData.texturesTable, currentTableName);

    pugi::xml_node options = currentNodeTable.append_child("Options");
    options.append_attribute("type").set_value(QSTR_TO_CSTR(newProjectData.texturesTable.rbTexturesType));
    options.append_attribute("genMipMaps").set_value(Util::String::boolToCstr(newProjectData.texturesTable.cbGenMipMaps));
    options.append_attribute("noUwrap").set_value(Util::String::boolToCstr(newProjectData.texturesTable.cbNoUwrap));
    options.append_attribute("noVwrap").set_value(Util::String::boolToCstr(newProjectData.texturesTable.cbNoVwrap));
    options.append_attribute("large").set_value(Util::String::boolToCstr(newProjectData.texturesTable.cbLarge));
    options.append_attribute("envMap").set_value(Util::String::boolToCstr(newProjectData.texturesTable.cbEnvMap));
    options.append_attribute("envMapValue").set_value(QSTR_TO_CSTR(newProjectData.texturesTable.leEnvMapTexture));

    // Characters tab

    currentTableName = CharactersTableName;

    currentNodeTable = fWriteTabGenericData(rootNode, newProjectData.charactersTable, currentTableName);

    options = currentNodeTable.append_child("Options");
    options.append_attribute("cellShading").set_value(Util::String::boolToCstr(newProjectData.charactersTable.cbCellShading));
    options.append_attribute("normals").set_value(Util::String::boolToCstr(newProjectData.charactersTable.cbNormals));
    options.append_attribute("standingPose").set_value(Util::String::boolToCstr(newProjectData.charactersTable.cbStandingPose));
    options.append_attribute("extractTRBSONCC").set_value(Util::String::boolToCstr(newProjectData.charactersTable.cbWithTRBS_ONCC));
    options.append_attribute("extractTRBSONCCValue").set_value(QSTR_TO_CSTR(newProjectData.charactersTable.leTRBS_ONCC));

    // Objects tab

    currentTableName = ObjectsTableName;

    currentNodeTable = fWriteTabGenericData(rootNode, newProjectData.objectsTable, currentTableName);

    options = currentNodeTable.append_child("Options");
    options.append_attribute("texture").set_value(Util::String::boolToCstr(newProjectData.objectsTable.cbTexture));
    options.append_attribute("textureValue").set_value(QSTR_TO_CSTR(newProjectData.objectsTable.leTextureName));
    options.append_attribute("withAnimation").set_value(Util::String::boolToCstr(newProjectData.objectsTable.cbWithAnimation));
    options.append_attribute("withAnimationValue").set_value(QSTR_TO_CSTR(newProjectData.objectsTable.leAnimationName));

    // Levels tab

    currentTableName = ObjectsTableName;

    currentNodeTable = fWriteTabGenericData(rootNode, newProjectData.levelsTable, currentTableName);

    options = currentNodeTable.append_child("Options");
    options.append_attribute("extractWithFiles").set_value(Util::String::boolToCstr(newProjectData.levelsTable.cbSpecificFilesLevels));
    options.append_attribute("extractWithFilesValue").set_value(QSTR_TO_CSTR(newProjectData.levelsTable.leSpecificFilesLevels));
    options.append_attribute("datFilename").set_value(Util::String::boolToCstr(newProjectData.levelsTable.cbDatLevels));
    options.append_attribute("datFilenameValue").set_value(QSTR_TO_CSTR(newProjectData.levelsTable.leTargetDatLevels));
    options.append_attribute("bnvSource").set_value(Util::String::boolToCstr(newProjectData.levelsTable.cbBnvLevels));
    options.append_attribute("bnvSourceValue").set_value(QSTR_TO_CSTR(newProjectData.levelsTable.leBnvLevels));
    options.append_attribute("generateGrids").set_value(Util::String::boolToCstr(newProjectData.levelsTable.cbGridsLevels));
    options.append_attribute("additionalSources").set_value(Util::String::boolToCstr(newProjectData.levelsTable.cbAdditionalSourcesLevels));
    options.append_attribute("additionalSourcesValue").set_value(QSTR_TO_CSTR(newProjectData.levelsTable.leAdditSourcesLevels));

    // Misc tab

    currentTableName = MiscTableName;

    fWriteTabGenericData(rootNode, newProjectData.miscTable, currentTableName);

    if(!doc.save_file(fileFullPath.toUtf8().constData(), PUGIXML_TEXT("\t"), pugi::format_default | pugi::format_write_bom, pugi::xml_encoding::encoding_utf8)){
        throw std::runtime_error("An error ocurred while trying to save the project file. Please try another path.");
    }

}
