unit _Extensions;

interface

uses
  _FileTypes, _TreeElement;

type
  TExtension = class(TTreeElement)
      function GetChildCount: Integer; override;
      function GetChild(ID: Integer): TTreeElement; override;
      function GetCaption: String; override;
    private
      FExt: String;
      FFiles: array of Integer;
    public
      constructor Create(ConnectionID: Integer; Ext: String); virtual;
      procedure InitList;
      property Ext: String read FExt;
  end;

  TExtensions = class(TTreeElement)
      function GetChildCount: Integer; override;
      function GetChild(ID: Integer): TTreeElement; override;
      function GetCaption: String; override;
    private
      FExtensions: array of TExtension;
    public
      constructor Create(DataAccess: TObject; ConnectionID: Integer);
      function GetExt(Ext: String): TExtension;
      procedure AddExt(Ext: String);
  end;



implementation

uses
  Classes, StrUtils, SysUtils, ConnectionManager, TypeDefs, DataAccess, _MetaManager;

{ TExtension }

constructor TExtension.Create(ConnectionID: Integer; Ext: String);
begin
  FConnectionID := ConnectionID;
  FExt := Ext;
end;

function TExtension.GetCaption: String;
begin
  Result := FExt + '[' + IntToStr(GetChildCount) + ']';
end;

function TExtension.GetChild(ID: Integer): TTreeElement;
var
  Meta: TMetaManager;
begin
  Meta := ConManager.Connection[FConnectionID].MetaData;
  Meta.InitFile(FFiles[ID]);
  Result := Meta.FileById[FFiles[ID]];
end;

function TExtension.GetChildCount: Integer;
begin
  Result := Length(FFiles);
end;

procedure TExtension.InitList;
var
  files: TStrings;
  i: Integer;
  fid: Integer;
  finfo: TFileInfo;
begin
  files := TStringList.Create;
  files := ConManager.Connection[FConnectionID].GetFilesList(FExt, '', False, ST_NameAsc);
  if files.Count > 0 then
  begin
    for i := 0 to files.Count - 1 do
    begin
      fid := StrToInt(MidStr(files.Strings[i], 1, 5));
      finfo := ConManager.Connection[FConnectionID].GetFileInfo(fid);
      if Length(finfo.Name) > 0 then
      begin
        SetLength(FFiles, Length(FFiles) + 1);
        FFiles[High(FFiles)] := fid;
      end;
    end;
  end;
  files.Free;
end;


{ TExtensions }

function TExtensions.GetCaption: String;
begin
  Result := '';
end;

function TExtensions.GetChild(ID: Integer): TTreeElement;
begin
  FExtensions[ID].InitList;
  Result := FExtensions[ID];
end;

function TExtensions.GetChildCount: Integer;
begin
  Result := Length(FExtensions);
end;

constructor TExtensions.Create(DataAccess: TObject; ConnectionID: Integer);
var
  files: TStrings;
  i: Integer;
  fid: Integer;
  finfo: TFileInfo;
begin
  FConnectionID := ConnectionID;
  files := TStringList.Create;
  files := TDataAccess(DataAccess).GetFilesList('', '', False, ST_IDAsc);
  SetLength(FExtensions, 0);
  if files.Count > 0 then
  begin
    for i := 0 to files.Count - 1 do
    begin
      fid := StrToInt(MidStr(files.Strings[i], 1, 5));
      finfo := TDataAccess(DataAccess).GetFileInfo(fid);
      if Length(finfo.Name) > 0 then
        if not Assigned(GetExt(finfo.Extension)) then
          AddExt(finfo.Extension);
    end;
  end;
  files.Free;
end;


function TExtensions.GetExt(Ext: String): TExtension;
var
  i: Integer;
begin
  Result := nil;
  if Length(FExtensions) > 0 then
    for i := 0 to High(FExtensions) do
      if FExtensions[i].Ext = Ext then
      begin
        Result := FExtensions[i];
        Break;
      end;
end;

procedure TExtensions.AddExt(Ext: String);
var
  i: Integer;
begin
  SetLength(FExtensions, Length(FExtensions) + 1);
  for i := High(FExtensions) downto 1 do
  begin
    if FExtensions[i-1].GetCaption < Ext then
    begin
      FExtensions[i] := TExtension.Create(FConnectionID, Ext);
      Break;
    end
    else
      FExtensions[i] := FExtensions[i-1];
  end;
  if i = 0 then
    FExtensions[0] := TExtension.Create(FConnectionID, Ext);
end;



end.
