package net.oni2.aeinstaller.backend;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.HashMap;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.io.xml.StaxDriver;

/**
 * Manages and stores programm settings
 * 
 * @author Christian Illy
 */
public class Settings implements Serializable {

	private static final long serialVersionUID = 8067725289301601179L;

	/**
	 * @author Christian Illy
	 */
	public enum Platform {
		/**
		 * Running Windows
		 */
		WIN,
		/**
		 * Running MacOS
		 */
		MACOS,
		/**
		 * Running a Linux
		 */
		LINUX,
		/**
		 * Unknown OS
		 */
		UNKNOWN
	}

	private static Settings instance = new Settings();

	private static boolean debugRun = false;

	private HashMap<String, Object> prefs = new HashMap<String, Object>();

	private boolean printNamesNotInMap = false;

	/**
	 * Get the singleton instance
	 * 
	 * @return Singleton instance
	 */
	public static Settings getInstance() {
		return instance;
	}

	/**
	 * @param debug
	 *            Debug mode
	 */
	public static void setDebug(boolean debug) {
		debugRun = debug;
	}

	/**
	 * @return Is debug run
	 */
	public static boolean getDebug() {
		return debugRun;
	}

	/**
	 * @return The operating system running on
	 */
	public static Platform getPlatform() {
		String os = System.getProperty("os.name").toLowerCase();
		if (os.startsWith("win"))
			return Platform.WIN;
		if (os.startsWith("linux"))
			return Platform.LINUX;
		if (os.startsWith("mac"))
			return Platform.MACOS;
		return Platform.UNKNOWN;
	}

	/**
	 * Get the Jar path
	 * 
	 * @return Path
	 */
	public static String getJarPath() {
		String jarPath = Settings.class.getProtectionDomain().getCodeSource()
				.getLocation().getPath();
		String decodedPath = null;
		try {
			decodedPath = URLDecoder.decode(jarPath, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		return new File(decodedPath).getParentFile().getPath() + "/";
	}

	/**
	 * Get the preferences path
	 * 
	 * @return Path
	 */
	public static String getPrefsPath() {
		return getJarPath();
	}

	/**
	 * Get the path to store downloaded files
	 * 
	 * @return Download path
	 */
	public static String getDownloadPath() {
		return getTempPath() + "oni_aei/";
	}

	/**
	 * Get the path to store game information data
	 * 
	 * @return Data path
	 */
	public static String getDataPath() {
		return getJarPath() + "mods/";
	}

	/**
	 * Get the systems temp-path
	 * 
	 * @return Path
	 */
	public static String getTempPath() {
		return new File(System.getProperty("java.io.tmpdir")).getPath() + "/";
	}

	/**
	 * @return Mod Depot cache filename
	 */
	public static String getDepotCacheFilename() {
		return Settings.getPrefsPath() + "ModDepotCache.xml";
	}

	private static String getSettingsFilename() {
		return Settings.getPrefsPath() + "AEI-Settings.xml";
	}

	private static XStream getXStream() {
		XStream xs = new XStream(new StaxDriver());
		xs.alias("Settings", Settings.class);
		return xs;
	}

	/**
	 * Serializes the settings to disk
	 */
	public void serializeToFile() {
		try {
			FileOutputStream fos = new FileOutputStream(getSettingsFilename());
			XStream xs = getXStream();
			xs.toXML(this, fos);
			fos.close();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Deserializes the settings from disk
	 */
	public static void deserializeFromFile() {
		try {
			FileInputStream fis = new FileInputStream(getSettingsFilename());
			XStream xs = getXStream();
			Object obj = xs.fromXML(fis);
			if (obj instanceof Settings)
				instance = (Settings) obj;
			fis.close();
		} catch (FileNotFoundException e) {
		} catch (IOException e) {
		}
	}

	/**
	 * Put a string value
	 * 
	 * @param key
	 *            Key for value
	 * @param value
	 *            Value
	 */
	public void put(String key, String value) {
		prefs.put(key, value);
	}

	/**
	 * Put a boolean value
	 * 
	 * @param key
	 *            Key for value
	 * @param value
	 *            Value
	 */
	public void put(String key, boolean value) {
		prefs.put(key, value);
	}

	/**
	 * Put a int value
	 * 
	 * @param key
	 *            Key for value
	 * @param value
	 *            Value
	 */
	public void put(String key, int value) {
		prefs.put(key, value);
	}

	/**
	 * Get a string value
	 * 
	 * @param key
	 *            Key for value
	 * @param def
	 *            Default return value if key does not exist
	 * @return Value
	 */
	public String get(String key, String def) {
		if (prefs.containsKey(key)) {
			if (prefs.get(key) instanceof String)
				return (String) (prefs.get(key));
		}
		if (printNamesNotInMap)
			System.out.println("Settings: Key \"" + key
					+ "\" not in Map, defaulting to \"" + def + "\".");
		return def;
	}

	/**
	 * Get a boolean value
	 * 
	 * @param key
	 *            Key for value
	 * @param def
	 *            Default return value if key does not exist
	 * @return Value
	 */
	public Boolean get(String key, Boolean def) {
		if (prefs.containsKey(key)) {
			if (prefs.get(key) instanceof Boolean)
				return (Boolean) (prefs.get(key));
		}
		if (printNamesNotInMap)
			System.out.println("Settings: Key \"" + key
					+ "\" not in Map, defaulting to \"" + def + "\".");
		return def;
	}

	/**
	 * Get a int value
	 * 
	 * @param key
	 *            Key for value
	 * @param def
	 *            Default return value if key does not exist
	 * @return Value
	 */
	public int get(String key, int def) {
		if (prefs.containsKey(key)) {
			if (prefs.get(key) instanceof Integer)
				return (Integer) (prefs.get(key));
		}
		if (printNamesNotInMap)
			System.out.println("Settings: Key \"" + key
					+ "\" not in Map, defaulting to \"" + def + "\".");
		return def;
	}

	/**
	 * Remove a value
	 * 
	 * @param key
	 *            Key to value to remove
	 */
	public void removeValue(String key) {
		prefs.remove(key);
	}

}
