package net.oni2.aeinstaller.gui.modtable;

import java.io.File;
import java.util.HashSet;
import java.util.ResourceBundle;
import java.util.TreeSet;
import java.util.Vector;

import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;

import net.oni2.aeinstaller.backend.mods.Mod;
import net.oni2.aeinstaller.backend.mods.ModManager;

/**
 * @author Christian Illy
 */
public class ModTableModel extends AbstractTableModel {

	private static final long serialVersionUID = -8278155705802697354L;

	private ResourceBundle bundle = ResourceBundle.getBundle(getClass()
			.getName());

	private Vector<Mod> items = new Vector<Mod>();
	private Vector<Boolean> install = new Vector<Boolean>();

	private HashSet<DownloadSizeListener> listeners = new HashSet<DownloadSizeListener>();

	/**
	 * Create a new model
	 */
	public ModTableModel() {
		reloadData();
	}

	@Override
	public Object getValueAt(int row, int col) {
		Mod mod = items.get(row);
		switch (col) {
			case -1:
				return mod;
			case 0:
				return install.get(row);
			case 1:
				return mod.getName();
			case 2:
				return mod.getPackageNumberString();
		}
		return null;
	}

	@Override
	public String getColumnName(int col) {
		switch (col) {
			case 0:
				return bundle.getString("mod.install");
			case 1:
				return bundle.getString("mod.name");
			case 2:
				return bundle.getString("mod.package_number");
		}
		return null;
	}

	@Override
	public int getRowCount() {
		return items.size();
	}

	@Override
	public int getColumnCount() {
		return 3;
	}

	@Override
	public Class<?> getColumnClass(int col) {
		switch (col) {
			case 0:
				return Boolean.class;
			case 1:
				return String.class;
			case 2:
				return String.class;
		}
		return null;
	}

	/**
	 * Set the constraints on the columns size for the given column
	 * 
	 * @param colNum
	 *            Column number
	 * @param col
	 *            Column object
	 */
	public void setColumnConstraints(int colNum, TableColumn col) {
		int w;
		switch (colNum) {
			case 0:
				w = 50;
				col.setPreferredWidth(w);
				col.setMinWidth(w);
				col.setMaxWidth(w);
				break;
			case 1:
				col.setPreferredWidth(150);
				break;
			case 2:
				w = 55;
				col.setPreferredWidth(w);
				col.setMinWidth(w);
				col.setMaxWidth(w);
				break;
		}
	}

	/**
	 * Reload the nodes data after an update to the cache
	 */
	public void reloadData() {
		items.clear();
		items.addAll(ModManager.getInstance().getMods());
		revertSelection();
	}

	/**
	 * Revert the selection to the mods that are currently installed
	 */
	public void revertSelection() {
		install.clear();
		for (int i = 0; i < items.size(); i++) {
			install.add(i, ModManager.getInstance()
					.isModInstalled(items.get(i)));
		}
		fireTableDataChanged();
	}

	/**
	 * Reload the selection after a config was loaded
	 * 
	 * @param config
	 *            Config to load
	 */
	public void reloadSelection(File config) {
		Vector<Integer> selected = ModManager.getInstance().loadModSelection(
				config);
		install.clear();
		for (int i = 0; i < items.size(); i++) {
			install.add(i, selected.contains(items.get(i).getPackageNumber()));
		}
		fireTableDataChanged();
	}

	/**
	 * Get the items vector
	 * 
	 * @return Items
	 */
	public Vector<Mod> getItems() {
		return items;
	}

	/**
	 * @return Mods selected for installation
	 */
	public TreeSet<Mod> getSelectedMods() {
		TreeSet<Mod> res = new TreeSet<Mod>();
		for (int i = 0; i < items.size(); i++) {
			if (install.get(i))
				res.add(items.get(i));
		}
		return res;
	}

	@Override
	public boolean isCellEditable(int rowIndex, int columnIndex) {
		return columnIndex == 0;
	}

	@Override
	public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
		super.setValueAt(aValue, rowIndex, columnIndex);
		if (columnIndex == 0) {
			install.set(rowIndex, (Boolean) aValue);

			int size = 0;
			for (int i = 0; i < items.size(); i++) {
				if (install.get(i)) {
					Mod m = items.get(i);
					if (!m.isLocalAvailable())
						size += m.getZipSize();
				}
			}
			for (DownloadSizeListener dsl : listeners)
				dsl.downloadSizeChanged(size);
		}
	}

	/**
	 * @param lis
	 *            Listener to receive download size changed events
	 */
	public void addDownloadSizeListener(DownloadSizeListener lis) {
		listeners.add(lis);
	}

	/**
	 * @param lis
	 *            Listener to no longer receive download size changed events
	 */
	public void removeDownloadSizeListener(DownloadSizeListener lis) {
		listeners.remove(lis);
	}

}
