package net.oni2.aeinstaller.backend.oni;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Map;
import java.util.Vector;

import net.oni2.aeinstaller.backend.Paths;
import net.oni2.aeinstaller.backend.AppExecution;
import net.oni2.aeinstaller.backend.Settings;
import net.oni2.aeinstaller.backend.Settings.Architecture;
import net.oni2.aeinstaller.backend.Settings.Platform;
import net.oni2.aeinstaller.backend.WinRegistry;

/**
 * @author Christian Illy
 */
public class OniSplit {

	/**
	 * @return is a .NET implementation installed?
	 */
	public static boolean isDotNETInstalled() {
		switch (Settings.getPlatform()) {
			case WIN:
				try {
					int view = WinRegistry.KEY_WOW64_32KEY;
					if (Settings.getArchitecture() == Architecture.AMD64)
						view = WinRegistry.KEY_WOW64_64KEY;

					Map<String, String> m = WinRegistry
							.readStringValues(
									WinRegistry.HKEY_LOCAL_MACHINE,
									"Software\\Microsoft\\NET Framework Setup\\NDP\\v2.0.50727",
									view);
					return m != null;
				} catch (IllegalArgumentException e) {
					e.printStackTrace();
				} catch (IllegalAccessException e) {
					e.printStackTrace();
				} catch (InvocationTargetException e) {
					e.printStackTrace();
				} catch (Exception e) {
					if (!e.getMessage()
							.equals("Registry access not supported (not a Windows OS?)."))
						e.printStackTrace();
				}
				return false;
			case MACOS:
			case LINUX:
				Vector<String> cmd = new Vector<String>();
				cmd.add("which");
				cmd.add("mono");
				Vector<String> res = null;
				try {
					res = AppExecution.executeAndWait(cmd);
				} catch (IOException e) {
					e.printStackTrace();
				}
				if (res != null) {
					if (res.get(0).startsWith("/")
							&& res.get(0).endsWith("mono")) {
						return true;
					}
				}
				return false;
			default:
				return false;
		}
	}

	/**
	 * @return Is Onisplit installed?
	 */
	public static boolean isOniSplitInstalled() {
		return getProgramFile().exists();
	}

	/**
	 * Export given dat-file to target folder
	 * 
	 * @param targetFolder
	 *            Target folder
	 * @param input
	 *            Dat file
	 * @return OniSplit output
	 */
	public static Vector<String> export(File targetFolder, File input) {
		if (!targetFolder.exists())
			targetFolder.mkdir();

		Vector<String> cmdLine = getProgramInvocation();
		cmdLine.add("-export");
		cmdLine.add(targetFolder.getPath());
		cmdLine.add(input.getPath());
		Vector<String> res = null;
		try {
			res = AppExecution.executeAndWait(cmdLine);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return res;
	}

	/**
	 * Import given folder to a .dat-file
	 * 
	 * @param sourceFolders
	 *            Folders containing .oni-files
	 * @param targetFile
	 *            Target .dat-file
	 * @return OniSplit output
	 */
	public static Vector<String> importLevel(Vector<File> sourceFolders,
			File targetFile) {
		Vector<String> cmdLine = getProgramInvocation();
		cmdLine.add(getImportParam());
		for (File f : sourceFolders)
			cmdLine.add(f.getPath());
		cmdLine.add(targetFile.getPath());
		Vector<String> res = null;
		try {
			res = AppExecution.executeAndWait(cmdLine);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return res;
	}

	/**
	 * Pack a level to a .dat-file. More powerful variant which allows
	 * specifying single .oni/.dat files
	 * 
	 * @param sourceFoldersFiles
	 *            Folders (for recursive .oni import) or files (.dat and single
	 *            .oni) to import
	 * @param targetFile
	 *            Target .dat-file
	 * @return OniSplit output
	 */
	public static Vector<String> packLevel(Vector<File> sourceFoldersFiles,
			File targetFile) {
		Vector<String> cmdLine = getProgramInvocation();
		cmdLine.add(getPackParam());
		cmdLine.add(getPackTypeParam());
		cmdLine.add("-out");
		cmdLine.add(targetFile.getPath());
		for (File f : sourceFoldersFiles)
			cmdLine.add(f.getPath());
		Vector<String> res = null;
		try {
			res = AppExecution.executeAndWait(cmdLine);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return res;
	}

	/**
	 * Move files from one location to another using OniSplit so relations are
	 * handled
	 * 
	 * @param targetFolder
	 *            Target folder for files
	 * @param input
	 *            Files to move, can contain wildcards
	 * @param moveParameter
	 *            e.g. overwrite, delete
	 * @return OniSplit output
	 */
	public static Vector<String> move(File targetFolder, String input,
			String moveParameter) {
		if (!targetFolder.exists())
			targetFolder.mkdir();

		Vector<String> cmdLine = getProgramInvocation();
		cmdLine.add("-move"
				+ (moveParameter != null ? ":" + moveParameter : ""));
		cmdLine.add(targetFolder.getPath());
		cmdLine.add(input);
		Vector<String> res = null;
		try {
			res = AppExecution.executeAndWait(cmdLine);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return res;
	}

	private static String getImportParam() {
		if (Settings.getPlatform() == Platform.MACOS)
			return "-import:sep";
		else
			return "-import:nosep";
	}

	private static String getPackParam() {
		return "pack";
	}

	private static String getPackTypeParam() {
		if (Settings.getPlatform() == Platform.MACOS)
			return "-type:macintel";
		else
			return "-type:pc";
	}

	private static Vector<String> getProgramInvocation() {
		Vector<String> res = new Vector<String>();
		if (Settings.getPlatform() != Platform.WIN)
			res.add("mono");
		res.add(getProgramFile().getPath());
		return res;
	}

	private static File getProgramFile() {
		return new File(new File(Paths.getEditionBasePath(), "Tools"),
				"Onisplit.exe");
	}
}
