package net.oni2.aeinstaller.gui.modtable;

import java.awt.Desktop;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.ResourceBundle;
import java.util.TreeSet;

import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.JViewport;
import javax.swing.ListSelectionModel;
import javax.swing.RowSorter;
import javax.swing.SortOrder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.table.TableRowSorter;

import net.oni2.aeinstaller.backend.mods.Mod;
import net.oni2.aeinstaller.backend.mods.Type;

/**
 * @author Christian Illy
 */
public class ModTable extends JTable {
	private static final long serialVersionUID = 1L;

	private ResourceBundle bundle = ResourceBundle
			.getBundle("net.oni2.aeinstaller.localization.ModTable");

	private HashSet<ModSelectionListener> modSelListeners = new HashSet<ModSelectionListener>();

	private ModTableModel model;
	private TableRowSorter<ModTableModel> sorter;

	/**
	 * Create a new ModTable
	 */
	public ModTable() {
		super();

		setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		getSelectionModel().addListSelectionListener(this);
		addMouseListener(new MouseEventHandler());
		addKeyListener(new KeyEventHandler());
		// To get checkbox-cells with background of row
		((JComponent) getDefaultRenderer(Boolean.class)).setOpaque(true);

		model = new ModTableModel();

		setModel(model);

		sorter = new TableRowSorter<ModTableModel>(model);
		setRowSorter(sorter);

		setFilter(null, 0);

		List<RowSorter.SortKey> sortKeys = new ArrayList<RowSorter.SortKey>();
		sortKeys.add(new RowSorter.SortKey(1, SortOrder.ASCENDING));
		sorter.setSortKeys(sortKeys);

		for (int i = 0; i < model.getColumnCount(); i++) {
			model.setColumnConstraints(i, getColumnModel().getColumn(i));
		}
	}

	@Override
	public String getToolTipText(MouseEvent e) {
		int r = rowAtPoint(e.getPoint());
		int c = columnAtPoint(e.getPoint());
		if (r >= 0 && r < getRowCount()) {
			int modelCol = convertColumnIndexToModel(c);
			if (modelCol == 4) {
				final Mod mod = (Mod) getValueAt(r, -1);

				String tt = "<html>";
				tt += String.format("%s: %s<br>",
						bundle.getString("state.installed"),
						bundle.getString((mod.isInstalled() ? "yes" : "no")));
				tt += String.format(
						"%s: %s<br>",
						bundle.getString("state.updatable"),
						bundle.getString((mod.isLocalAvailable()
								&& mod.isNewerAvailable() ? "yes" : "no")));
				tt += String.format("%s: %s</html>", bundle
						.getString("state.downloaded"), bundle.getString((mod
						.isLocalAvailable() ? "yes" : "no")));
				return tt;
			}
		}
		return super.getToolTipText(e);
	}

	/**
	 * @param listener
	 *            Listener to add
	 */
	public void addModSelectionListener(ModSelectionListener listener) {
		modSelListeners.add(listener);
	}

	/**
	 * @param listener
	 *            Listener to remove
	 */
	public void removeModSelectionListener(ModSelectionListener listener) {
		modSelListeners.remove(listener);
	}

	private void notifyModSelectionListeners(Mod m) {
		for (ModSelectionListener l : modSelListeners) {
			l.modSelectionChanged(this, m);
		}
	}

	/**
	 * @param listener
	 *            Listener to add
	 */
	public void addDownloadSizeListener(DownloadSizeListener listener) {
		model.addDownloadSizeListener(listener);
	}

	/**
	 * @param listener
	 *            Listener to remove
	 */
	public void removeDownloadSizeListener(DownloadSizeListener listener) {
		model.removeDownloadSizeListener(listener);
	}

	/**
	 * Reload the nodes data after an update to the cache
	 */
	public void reloadData() {
		model.reloadData();
	}

	/**
	 * Revert the selection to the mods that are currently installed
	 */
	public void revertSelection() {
		model.revertSelection();
	}

	/**
	 * Reload the selection after a config was loaded
	 * 
	 * @param config
	 *            Config to load
	 */
	public void reloadSelection(File config) {
		model.reloadSelection(config);
	}

	/**
	 * @return Mods selected for installation
	 */
	public TreeSet<Mod> getSelectedMods() {
		return model.getSelectedMods();
	}

	/**
	 * @param type
	 *            Type of mods to show (null for all)
	 * @param downloadState
	 *            Show only: 0 = all, 1 = online, 2 = downloaded
	 */
	public void setFilter(Type type, int downloadState) {
		sorter.setRowFilter(new ModTableFilter(type, downloadState));
	}

	@Override
	public void valueChanged(ListSelectionEvent e) {
		super.valueChanged(e);
		int viewRow = getSelectedRow();
		if (viewRow < 0) {
			notifyModSelectionListeners(null);
		} else {
			Mod mod = (Mod) getValueAt(viewRow, -1);
			notifyModSelectionListeners(mod);
		}
	}

	private class MouseEventHandler extends MouseAdapter {
		private void mouseEventProcessing(MouseEvent e) {
			int r = rowAtPoint(e.getPoint());
			if (r >= 0 && r < getRowCount())
				setRowSelectionInterval(r, r);
			else
				clearSelection();

			int rowindex = getSelectedRow();
			if (rowindex >= 0) {
				if (e.isPopupTrigger() && e.getComponent() instanceof JTable) {
					final Mod mod = (Mod) getValueAt(rowindex, -1);

					JPopupMenu popup = new JPopupMenu();

					if (mod.isLocalAvailable()) {
						// Open package folder item
						JMenuItem openModFolder = new JMenuItem(
								bundle.getString("openModFolder.text"));
						openModFolder.addActionListener(new ActionListener() {
							@Override
							public void actionPerformed(ActionEvent arg0) {
								try {
									Desktop.getDesktop().open(
											mod.getLocalPath());
								} catch (IOException e) {
									e.printStackTrace();
								}
							}
						});
						popup.add(openModFolder);
					}

					if (mod.getUrl() != null) {
						// Open Depot page item
						JMenuItem openDepotPage = new JMenuItem(
								bundle.getString("openDepotPage.text"));
						openDepotPage.addActionListener(new ActionListener() {
							@Override
							public void actionPerformed(ActionEvent arg0) {
								try {
									Desktop.getDesktop().browse(mod.getUrl());
								} catch (IOException e) {
									e.printStackTrace();
								}
							}
						});
						popup.add(openDepotPage);
					}

					if (popup.getSubElements().length > 0)
						popup.show(e.getComponent(), e.getX(), e.getY());
				}
			}
		}

		@Override
		public void mousePressed(MouseEvent e) {
			mouseEventProcessing(e);
		}

		@Override
		public void mouseReleased(MouseEvent e) {
			mouseEventProcessing(e);
		}
	}

	private class KeyEventHandler extends KeyAdapter {
		@Override
		public void keyTyped(KeyEvent e) {
			super.keyTyped(e);

			if (e.getModifiers() == 0) {
				String key = String.valueOf(e.getKeyChar()).toLowerCase();
				for (int i = 0; i < getRowCount(); i++) {
					Mod m = (Mod) getValueAt(i, -1);
					if (m.getName().toLowerCase().startsWith(key)) {
						setRowSelectionInterval(i, i);
						JViewport viewport = (JViewport) getParent();
						Rectangle rect = getCellRect(i, 0,
								true);
						Rectangle r2 = viewport.getVisibleRect();
						scrollRectToVisible(new Rectangle(rect.x, rect.y,
								(int) r2.getWidth(), (int) r2.getHeight()));
						break;
					}
				}
			}
		}
	}
}
