package net.oni2.aeinstaller.gui.modtable;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.ResourceBundle;
import java.util.TreeSet;
import java.util.Vector;

import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;

import net.oni2.aeinstaller.backend.packages.Package;
import net.oni2.aeinstaller.backend.packages.PackageManager;

/**
 * @author Christian Illy
 */
public class ModTableModel extends AbstractTableModel {

	private static final long serialVersionUID = -8278155705802697354L;

	private ResourceBundle bundle = ResourceBundle
			.getBundle("net.oni2.aeinstaller.localization.ModTable");

	private Vector<Package> items = new Vector<Package>();
	private Vector<Boolean> install = new Vector<Boolean>();

	private HashSet<DownloadSizeListener> listeners = new HashSet<DownloadSizeListener>();

	private SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

	/**
	 * Create a new model
	 */
	public ModTableModel() {
	}

	@Override
	public Object getValueAt(int row, int col) {
		Package mod = items.get(row);
		switch (col) {
			case -1:
				return mod;
			case 0:
				return install.get(row);
			case 1:
				return mod.getName();
			case 2:
				return mod.getPackageNumberString();
			case 3:
				return mod.getCreator();
			case 4:
				String res = "";
				res += (mod.isInstalled() ? "I" : "_");
				res += (mod.isLocalAvailable() && mod.isNewerAvailable() ? "U"
						: "_");
				res += (mod.isLocalAvailable() ? "D" : "_");
				return res;
			case 5:
				if (mod.getFile() != null)
					return sdf.format(new Date(
							mod.getFile().getTimestamp() * 1000));
				else
					return null;
		}
		return null;
	}

	@Override
	public String getColumnName(int col) {
		switch (col) {
			case 0:
				return bundle.getString("mod.install");
			case 1:
				return bundle.getString("mod.name");
			case 2:
				return bundle.getString("mod.package_number");
			case 3:
				return bundle.getString("mod.creator");
			case 4:
				return bundle.getString("mod.state");
			case 5:
				return bundle.getString("mod.date");
		}
		return null;
	}

	@Override
	public int getRowCount() {
		return items.size();
	}

	@Override
	public int getColumnCount() {
		return 6;
	}

	@Override
	public Class<?> getColumnClass(int col) {
		switch (col) {
			case 0:
				return Boolean.class;
			case 1:
				return String.class;
			case 2:
				return String.class;
			case 3:
				return String.class;
			case 4:
				return String.class;
			case 5:
				return String.class;
		}
		return null;
	}

	/**
	 * Set the constraints on the columns size for the given column
	 * 
	 * @param colNum
	 *            Column number
	 * @param col
	 *            Column object
	 */
	public void setColumnConstraints(int colNum, TableColumn col) {
		int w;
		switch (colNum) {
			case 0:
				w = 70;
				col.setPreferredWidth(w);
				col.setMinWidth(w);
				col.setMaxWidth(w);
				break;
			case 1:
				col.setPreferredWidth(150);
				break;
			case 2:
				w = 60;
				col.setPreferredWidth(w);
				col.setMinWidth(w);
				col.setMaxWidth(w);
				break;
			case 3:
				col.setPreferredWidth(90);
				break;
			case 4:
				w = 60;
				col.setPreferredWidth(w);
				col.setMinWidth(w);
				col.setMaxWidth(w);
				break;
			case 5:
				w = 115;
				col.setPreferredWidth(w);
				col.setMinWidth(w);
				col.setMaxWidth(w);
				break;
		}
	}

	/**
	 * Reload the nodes data after an update to the cache
	 */
	public void reloadData() {
		items.clear();
		items.addAll(PackageManager.getInstance().getModsValidAndNotCore());
		revertSelection();
	}

	/**
	 * Revert the selection to the mods that are currently installed
	 */
	public void revertSelection() {
		install.clear();
		for (int i = 0; i < items.size(); i++) {
			install.add(i, items.get(i).isInstalled());
		}
		notifyDownloadSize(0);
		fireTableDataChanged();
	}

	/**
	 * Reload the selection after a config was loaded
	 * 
	 * @param config
	 *            Config to load
	 */
	public void reloadSelection(File config) {
		Vector<Integer> selected = PackageManager.getInstance()
				.loadModSelection(config);
		install.clear();
		for (int i = 0; i < items.size(); i++) {
			install.add(i, selected.contains(items.get(i).getPackageNumber()));
		}
		fireTableDataChanged();
	}

	/**
	 * Get the items vector
	 * 
	 * @return Items
	 */
	public Vector<Package> getItems() {
		return items;
	}

	/**
	 * @return Mods selected for installation
	 */
	public TreeSet<Package> getSelectedMods() {
		TreeSet<Package> res = new TreeSet<Package>();
		for (int i = 0; i < items.size(); i++) {
			if (install.get(i))
				res.add(items.get(i));
		}
		return res;
	}

	@Override
	public boolean isCellEditable(int rowIndex, int columnIndex) {
		return columnIndex == 0;
	}

	private void notifyDownloadSize(int size) {
		for (DownloadSizeListener dsl : listeners)
			dsl.downloadSizeChanged(size);
	}

	@Override
	public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
		super.setValueAt(aValue, rowIndex, columnIndex);
		if (columnIndex == 0) {
			install.set(rowIndex, (Boolean) aValue);

			int size = 0;
			for (int i = 0; i < items.size(); i++) {
				if (install.get(i)) {
					Package m = items.get(i);
					if (!m.isLocalAvailable())
						size += m.getZipSize();
				}
			}
			notifyDownloadSize(size);
		}
	}

	/**
	 * @param lis
	 *            Listener to receive download size changed events
	 */
	public void addDownloadSizeListener(DownloadSizeListener lis) {
		listeners.add(lis);
	}

	/**
	 * @param lis
	 *            Listener to no longer receive download size changed events
	 */
	public void removeDownloadSizeListener(DownloadSizeListener lis) {
		listeners.remove(lis);
	}

}
