package net.oni2.aeinstaller.backend.oni;

import java.io.File;
import java.io.IOException;
import java.util.Vector;

import net.oni2.aeinstaller.backend.CaseInsensitiveFile;
import net.oni2.aeinstaller.backend.DotNet;
import net.oni2.aeinstaller.backend.Paths;
import net.oni2.applicationinvoker.AppExecution;
import net.oni2.applicationinvoker.AppExecutionResult;
import net.oni2.settingsmanager.Settings;
import net.oni2.settingsmanager.Settings.Platform;

/**
 * @author Christian Illy
 */
public class OniSplit {

	/**
	 * @return Is Onisplit installed?
	 */
	public static boolean isOniSplitInstalled() {
		return getProgramFile().exists();
	}

	/**
	 * Export given dat-file to target folder
	 * 
	 * @param targetFolder
	 *            Target folder
	 * @param input
	 *            Dat file
	 * @return OniSplit output
	 */
	public static AppExecutionResult export(File targetFolder, File input) {
		return export(targetFolder, input, null);
	}

	/**
	 * Export named resources from given dat-file to target folder
	 * 
	 * @param targetFolder
	 *            Target folder
	 * @param input
	 *            Dat file
	 * @param patterns
	 *            Filename patterns for files to export
	 * @return OniSplit output
	 */
	public static AppExecutionResult export(File targetFolder, File input,
			Vector<String> patterns) {
		if (!targetFolder.exists())
			targetFolder.mkdir();

		Vector<String> cmdLine = getProgramInvocation();
		if (patterns == null)
			cmdLine.add("-export");
		else {
			for (String p : patterns)
				cmdLine.add("-export:" + p);
		}
		cmdLine.add(targetFolder.getPath());
		cmdLine.add(input.getPath());
		AppExecutionResult res = null;
		try {
			res = AppExecution.executeAndWait(cmdLine);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return res;
	}

	/**
	 * Import given folder to a .dat-file
	 * 
	 * @param sourceFolders
	 *            Folders containing .oni-files
	 * @param targetFile
	 *            Target .dat-file
	 * @return OniSplit output
	 */
	public static AppExecutionResult importLevel(Vector<File> sourceFolders,
			File targetFile) {
		Vector<String> cmdLine = getProgramInvocation();
		cmdLine.add(getImportParam());
		for (File f : sourceFolders)
			cmdLine.add(f.getPath());
		cmdLine.add(targetFile.getPath());
		AppExecutionResult res = null;
		try {
			res = AppExecution.executeAndWait(cmdLine);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return res;
	}

	/**
	 * Pack a level to a .dat-file. More powerful variant which allows
	 * specifying single .oni/.dat files
	 * 
	 * @param sourceFoldersFiles
	 *            Folders (for recursive .oni import) or files (.dat and single
	 *            .oni) to import
	 * @param targetFile
	 *            Target .dat-file
	 * @return OniSplit output
	 */
	public static AppExecutionResult packLevel(Vector<File> sourceFoldersFiles,
			File targetFile) {
		Vector<String> cmdLine = getProgramInvocation();
		cmdLine.add(getPackParam());
		cmdLine.add(getPackTypeParam());
		cmdLine.add("-out");
		cmdLine.add(targetFile.getPath());
		for (File f : sourceFoldersFiles)
			cmdLine.add(f.getPath());
		AppExecutionResult res = null;
		try {
			res = AppExecution.executeAndWait(cmdLine);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return res;
	}

	/**
	 * Move files from one location to another using OniSplit so relations are
	 * handled
	 * 
	 * @param targetFolder
	 *            Target folder for files
	 * @param input
	 *            Files to move, can contain wildcards
	 * @param moveParameter
	 *            e.g. overwrite, delete
	 * @return OniSplit output
	 */
	public static AppExecutionResult move(File targetFolder, String input,
			String moveParameter) {
		if (!targetFolder.exists())
			targetFolder.mkdir();

		Vector<String> cmdLine = getProgramInvocation();
		cmdLine.add("-move"
				+ (moveParameter != null ? ":" + moveParameter : ""));
		cmdLine.add(targetFolder.getPath());
		cmdLine.add(input);
		AppExecutionResult res = null;
		try {
			res = AppExecution.executeAndWait(cmdLine);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return res;
	}

	/**
	 * Convert given .oni-files to XML and put them in target folder
	 * 
	 * @param targetFolder
	 *            Target folder
	 * @param inputFiles
	 *            .oni files
	 * @return OniSplit output
	 */
	public static AppExecutionResult convertOniToXML(File targetFolder,
			Vector<File> inputFiles) {
		if (!targetFolder.exists())
			targetFolder.mkdirs();

		Vector<String> cmdLine = getProgramInvocation();
		cmdLine.add("-extract:xml");
		cmdLine.add(targetFolder.getPath());
		for (File f : inputFiles) {
			cmdLine.add(f.getPath());
		}
		AppExecutionResult res = null;
		try {
			res = AppExecution.executeAndWait(cmdLine);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return res;
	}

	/**
	 * Convert given XML-files to .oni and put them in target folder
	 * 
	 * @param targetFolder
	 *            Target folder
	 * @param inputFiles
	 *            XML-files
	 * @return OniSplit output
	 */
	public static AppExecutionResult convertXMLtoOni(File targetFolder,
			Vector<File> inputFiles) {
		if (!targetFolder.exists())
			targetFolder.mkdirs();

		Vector<String> cmdLine = getProgramInvocation();
		cmdLine.add("-create");
		cmdLine.add(targetFolder.getPath());
		for (File f : inputFiles) {
			cmdLine.add(f.getPath());
		}
		AppExecutionResult res = null;
		try {
			res = AppExecution.executeAndWait(cmdLine);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return res;
	}

	private static String getImportParam() {
		if (Settings.getPlatform() == Platform.MACOS)
			return "-import:sep";
		else
			return "-import:nosep";
	}

	private static String getPackParam() {
		return "pack";
	}

	private static String getPackTypeParam() {
		if (Settings.getPlatform() == Platform.MACOS)
			return "-type:macintel";
		else
			return "-type:pc";
	}

	private static Vector<String> getProgramInvocation() {
		Vector<String> res = new Vector<String>();
		if (DotNet.getRuntimeExe().length() > 0)
			res.add(DotNet.getRuntimeExe());
		res.add(getProgramFile().getPath());
		return res;
	}

	private static File getProgramFile() {
		File toolsPath = CaseInsensitiveFile.getCaseInsensitiveFile(
				Paths.getEditionBasePath(), "Tools");
		return CaseInsensitiveFile.getCaseInsensitiveFile(toolsPath,
				"Onisplit.exe");
	}
}
