#include "soundpagefinal.h"
#include "ui_soundpagefinal.h"

SoundPageFinal::SoundPageFinal(QString soundsLocation, DropTableWidget *page2Table, Logger *myLogger, QHash<QString, QString> *commandMap, QWidget *parent) :
    QWizardPage(parent),
    ui(new Ui::soundpagefinal)
{
    ui->setupUi(this);
    this->soundsLocation=soundsLocation;
    this->page2Table=page2Table;
    this->myLogger=myLogger;
    this->commandMap=commandMap;

    this->xmlCommands = new QStringList();
    this->oniSplitCommands = new QStringList();
    this->myXmlProcessor = new XmlProcessor(this->myLogger,this->xmlCommands);
    this->myConverter = new Converter(this->myLogger,this->oniSplitCommands);

    connectSlots();
}

void SoundPageFinal::openSoundsFolder(){
    QString outputFolder;

    if(field("rbOther").toBool()){
        outputFolder=field("leOtherLocation").toString();
    }
    else{
        outputFolder=this->soundsLocation;
    }

    QDesktopServices::openUrl(QUrl("file:///"+outputFolder));
}

void SoundPageFinal::initializePage(){
    startProcessing();
}

void SoundPageFinal::startProcessing(){
    // Sample files names
    QString ambFile="OSBDsample_file.amb.xml";
    QString grpFile="OSBDsample_file.grp.xml";
    QString ambFileLocation=GlobalVars::VagoTemporaryDir+"/"+ambFile;
    QString grpFileLocation=GlobalVars::VagoTemporaryDir+"/"+grpFile;

    // Page 2 variables
    QString outputFolder;

    // Page 3 variables
    QString sphereRadious, minElapsedTime, maxElapsedTime, minVolumeDistance,
            maxVolumeDistance, minOcclusion, treshold, priority;
    QStringList flags;
    bool priorityLow = false, priorityNormal = false,
            priorityHigh = false, priorityHighest = false;

    bool interruptOnStop=false;
    bool playOnce=false;
    bool canPan=false;

    // Page 4 variables
    QString volume, minVolume, maxVolume, pitch, minPitch, weight, maxPitch,
            numberChannels;
    bool preventRepeat=false;
    bool stereo22=false, mono22=false, mono44Pc=false;

    // Get data page 2
    if(field("rbOther").toBool()){
        outputFolder=field("leOtherLocation").toString();
    }
    else{
        outputFolder=this->soundsLocation;
    }

    outputFolder = Util::insertQuotes(outputFolder); // for onisplit work correctly

    // Get data page 3

    priorityLow=field("rbPriorityLow").toBool();
    priorityNormal=field("rbPriorityNormal").toBool();
    priorityHigh=field("rbPriorityHigh").toBool();
    priorityHighest=field("rbPriorityHighest").toBool();

    if(priorityNormal){
        priority="Normal";
    }
    else if(priorityLow){
        priority="Low";
    }
    else if(priorityHigh){
        priority="High";
    }
    else if(priorityHighest){
        priority="Highest";
    }

    if(interruptOnStop){
        flags << "InterruptTracksOnStop";
    }

    if(playOnce){
        flags << "PlayOnce";
    }

    if(canPan){
        flags << "CanPan";
    }

    interruptOnStop=field("cbInterruptOnStop").toBool();
    playOnce=field("cbPlayOnce").toBool();
    canPan=field("cbCanPan").toBool();

    sphereRadious=Util::normalizeDecimalSeparator(field("leSphereRadious").toString());
    minElapsedTime=Util::normalizeDecimalSeparator(field("leMinElapsedTime").toString());
    maxElapsedTime=Util::normalizeDecimalSeparator(field("leMaxElapsedTime").toString());
    minVolumeDistance=Util::normalizeDecimalSeparator(field("leMinVolumeDistance").toString());
    maxVolumeDistance=Util::normalizeDecimalSeparator(field("leMaxVolumeDistance").toString());
    minOcclusion=Util::normalizeDecimalSeparator(field("leMinOcclusion").toString());
    treshold=Util::normalizeDecimalSeparator(field("leTreshold").toString());

    // Get data page 4

    volume=Util::normalizeDecimalSeparator(field("leVolume").toString());
    minVolume=Util::normalizeDecimalSeparator(field("leMinVolume").toString());
    maxVolume=Util::normalizeDecimalSeparator(field("leMaxVolume").toString());
    pitch=Util::normalizeDecimalSeparator(field("lePitch").toString());
    minPitch=Util::normalizeDecimalSeparator(field("leMinPitch").toString());
    maxPitch=Util::normalizeDecimalSeparator(field("leMaxPitch").toString());
    weight=Util::normalizeDecimalSeparator(field("leWeight").toString());

    preventRepeat=field("cbPreventRepeat").toBool();

    stereo22=field("rbStereo22").toBool();
    mono22=field("rbMono22").toBool();
    mono44Pc=field("rbMono44Pc").toBool();

    if(stereo22 || mono44Pc){
        numberChannels="2";
    }
    else if(mono22){
        numberChannels="1";
    }
    //######################################################### Starting xml processing

    // Clean tmp dir
    if(!Util::rmDir(GlobalVars::VagoTemporaryDir)){
        QString message="Couldn't clean vago temp dir for complete sound wizard. Aborting.\n"+GlobalVars::VagoTemporaryDir;
        this->myLogger->writeString(message);
        Util::showErrorLogPopUp(message);
        return;
    }

    if(!QDir(QDir::tempPath()).mkpath("VagoTemp")){
        QString message="Couldn't reconstruct vago temp dir for complete sound wizard. Aborting.\n"+GlobalVars::VagoTemporaryDir;
        this->myLogger->writeString(message);
        Util::showErrorLogPopUp(message);
    }

    // Copy sample xml files to tmp dir
    QFile::copy(":/new/sampleFiles/"+ambFile , ambFileLocation);
    QFile::copy(":/new/sampleFiles/"+grpFile , grpFileLocation);
    QFile::setPermissions(ambFileLocation, QFile::ReadOwner | QFile::WriteOwner); //remove read only attribute that come from resources
    QFile::setPermissions(grpFileLocation, QFile::ReadOwner | QFile::WriteOwner);

    (*this->xmlCommands) << "replaceall -element:Priority -value:"+priority+" -filename:"+ambFileLocation+" -nobackups"
                         << "replaceall -element:Flags -value:"+flags.join(" ")+" -filename:"+ambFileLocation+" -nobackups"
                         << "replaceall -element:SphereRadius -value:"+sphereRadious+" -filename:"+ambFileLocation+" -nobackups"
                         << "replaceall -element:Treshold -value:"+treshold+" -filename:"+ambFileLocation+" -nobackups"
                         << "replaceall -element:MinOcclusion -value:"+minOcclusion+" -filename:"+ambFileLocation+" -nobackups"
                         << "replaceall -parelement:ElapsedTime -element:Min -value:"+minElapsedTime+" -filename:"+ambFileLocation+" -nobackups"
                         << "replaceall -parelement:Distance -element:Max -value:"+maxVolumeDistance+" -filename:"+ambFileLocation+" -nobackups"
                         << "replaceall -parelement:Distance -element:Min -value:"+minVolumeDistance+" -filename:"+ambFileLocation+" -nobackups"
                         << "replaceall -parelement:SoundGroup -element:Volume -value:"+volume+" -filename:"+grpFileLocation+" -nobackups"
                         << "replaceall -parelement:SoundGroup -element:Pitch -value:"+pitch+" -filename:"+grpFileLocation+" -nobackups"
                         << "replaceall -parelement:SoundGroup -element:NumberOfChannels -value:"+numberChannels+" -filename:"+grpFileLocation+" -nobackups"
                         << "replaceall -parelement:Volume -element:Min -value:"+minVolume+" -filename:"+grpFileLocation+" -nobackups"
                         << "replaceall -parelement:Volume -element:Max -value:"+maxVolume+" -filename:"+grpFileLocation+" -nobackups"
                         << "replaceall -parelement:Pitch -element:Min -value:"+minPitch+" -filename:"+grpFileLocation+" -nobackups"
                         << "replaceall -parelement:Pitch -element:Max -value:"+maxPitch+" -filename:"+grpFileLocation+" -nobackups"
                         << "replaceall -element:Weight -value:"+weight+" -filename:"+grpFileLocation+" -nobackups";

    if(preventRepeat){
        (*this->xmlCommands) << "replaceall -parelement:SoundGroup -element:Flags -value:PreventRepeat -filename:"+grpFileLocation+" -nobackups";
    }

    myXmlProcessor->start();
    myXmlProcessor->wait(); // Wait until all xml is edited

    QString currFileName;
    QString currGrpFileLocation;
    QString currAmbFileLocation;

    for(int i=0; i<this->page2Table->rowCount(); i++){

        (*this->oniSplitCommands) << this->commandMap->value("general->XML->ONI")+" "+outputFolder+" "+Util::insertQuotes(this->page2Table->item(i,1)->text()); // add location of sound file to convert

        currFileName=this->page2Table->item(i,0)->text(); // get current file name
        currAmbFileLocation = QString(ambFileLocation).replace("sample_file",currFileName); // get the new files, filenames
        currGrpFileLocation = QString(grpFileLocation).replace("sample_file",currFileName);

        QFile::copy(ambFileLocation, currAmbFileLocation); // make a copy of the sample files that will be the real files
        QFile::copy(grpFileLocation, currGrpFileLocation);

        (*this->xmlCommands) << "replaceall -element:BaseTrack1 -value:"+currFileName+" -filename:"+currAmbFileLocation+" -nobackups" // process the xml
                             << "replaceall -element:Sound -value:"+currFileName+" -filename:"+currGrpFileLocation+" -nobackups";

        myXmlProcessor->start();
        myXmlProcessor->wait(); // Wait until all xml is edited
    }

    (*this->oniSplitCommands) << this->commandMap->value("general->XML->ONI")+" "+outputFolder+" "+Util::insertQuotes(GlobalVars::VagoTemporaryDir+"/*.xml");

    this->myConverter->start(); // finally process the onisplit commands
    this->myConverter->wait(); // wait for it to complete

    // Finally remove the sample files, since not needed anymore
    QFile(ambFileLocation).remove();
    QFile(grpFileLocation).remove();
}

void SoundPageFinal::catchXmlAndOSplitProcessingError(QString result, int numErrors){

    if(numErrors!=0){
        QString sNumErrors=QString::number(numErrors);
        if(numErrors>1){
            Util::showErrorLogPopUp(result+"\n This is the last of "+sNumErrors+" Errors.");
        }
        else{
            Util::showErrorLogPopUp(result);
        }
    }
}

void SoundPageFinal::connectSlots(){
    //This signal is for thread that is working setup the progress bar (make it visible and set it's min-max)
    connect(this->myXmlProcessor, SIGNAL(resultConversion(QString, int)), this, SLOT(catchXmlAndOSplitProcessingError(QString, int)));
    connect(this->myConverter, SIGNAL(resultConversion(QString, int)), this, SLOT(catchXmlAndOSplitProcessingError(QString, int)));
    connect(ui->lbComplete, SIGNAL(linkActivated(const QString & )), this, SLOT(openSoundsFolder()));
}

SoundPageFinal::~SoundPageFinal()
{
    delete this->xmlCommands;
    delete this->oniSplitCommands;
    delete this->myXmlProcessor;
    delete this->myConverter;
    delete ui;
}
