#include "packagewizard.h"

PackageWizard::PackageWizard(QString workspaceWizardLocation, QSettings *vagoSettings, Logger *myLogger)
{
    this->workspaceWizardLocation=workspaceWizardLocation;
    this->vagoSettings=vagoSettings;
    this->myLogger=myLogger;
    this->packagesLocation=this->workspaceWizardLocation+"/Packages";
}

int PackageWizard::exec(){
    QWizard myWizard;

    myWizard.setWindowIcon(QIcon(":/new/icons/package.png"));

    //Center and resize QWizard (http://www.thedazzlersinc.com/source/2012/06/04/qt-center-window-in-screen/)
    myWizard.resize(640,480);
    QRect position = myWizard.frameGeometry();
    position.moveCenter(QDesktopWidget().availableGeometry().center());
    myWizard.move(position.topLeft());
    //

    PackagePage2 *page2 = new PackagePage2(this->myLogger);
    PackagePage3 *page3 = new PackagePage3();
    PackagePage4 *page4 = new PackagePage4();
    PackagePageFinal *pageFinal = new PackagePageFinal(this->vagoSettings);

    myWizard.addPage(createIntroPage());
    myWizard.addPage(page2);
    myWizard.addPage(page3);
    myWizard.addPage(page4);
    myWizard.addPage(pageFinal);

    myWizard.setWindowTitle("AIE2 Package Creator");

    //If wizard finished with sucess
    if(myWizard.exec()){ //modal and wait for finalization
        createPackage(myWizard, page4);
    }

    return 0;
}

QWizardPage* PackageWizard::createIntroPage() {
    QWizardPage *page = new QWizardPage;
    page->setTitle("Introduction");

    QLabel *label = new QLabel("Welcome to the Anniversary Edition Installer 2 (AIE2) package"
                               " creator wizard.\n"
                               "This wizard will allow you to create in a few and simple steps a package for AIE2.");
    label->setWordWrap(true);

    QVBoxLayout *layout = new QVBoxLayout;
    layout->addWidget(label);
    page->setLayout(layout);

    return page;
}

void PackageWizard::createPackage(const QWizard &myWizard, PackagePage4 *page4){
    const QString aeVersion="2.0";

    //Get info page 2
    QString modName=myWizard.field("leModName").toString();
    QString authors=myWizard.field("leAuthors").toString();
    QString version=myWizard.field("leVersion").toString();
    QString description=myWizard.field("ptDescription").toString();
    QString packageNumber=myWizard.field("lePackageNumber").toString();
    bool bslReplace=myWizard.field("rbReplace").toBool();

    //Get info page 3
    QString dependentPackages=myWizard.field("leDependentPackages").toString();
    QString incompatiblePackages=myWizard.field("leIncompatiblePackages").toString();
    QString unlockLevels=myWizard.field("leUnlockLevels").toString();

    //Get info page 4
    const DropTableWidget *commonTable=page4->commonTable;
    const DropTableWidget *windowsTable=page4->windowsTable;
    const DropTableWidget *macTable=page4->macTable;

    //Get info from final page
    bool openFolder=myWizard.field("cbOpenFolder").toBool();
    bool createZip=myWizard.field("cbCreateZip").toBool();
    //Remember the final page choices to next time
    this->vagoSettings->setValue("PackageCreator/OpenFolder",openFolder);
    this->vagoSettings->setValue("PackageCreator/CreateZip",createZip);

    const QString packageName=packageNumber+Util::fullTrim(modName);

    // Start package creation...

    // Create Packages folder if it doesn't exist
    if(!QDir(this->packagesLocation).exists()){
        QDir().mkpath(this->packagesLocation);
    }

    QString modDir=this->packagesLocation+"/"+packageName+"/";

    QDir().mkdir(modDir);

    bool bslExist=false;

    if(commonTable->rowCount()>0){
        copyPackageFolders(commonTable,"common",modDir,bslExist);
    }

    if(windowsTable->rowCount()>0){
        copyPackageFolders(windowsTable,"win_only",modDir,bslExist);
    }

    if(macTable->rowCount()>0){
        copyPackageFolders(macTable,"mac_only",modDir,bslExist);
    }

    QFile *modInfo = new QFile(modDir+"Mod_Info.cfg");

    if (!modInfo->open(QIODevice::WriteOnly | QIODevice::Text)){ //open to write
        Util::showErrorPopUp("Couldn't create Mod_Info.cfg file.");
        myLogger->writeString("Couldn't create Mod_Info.cfg file when creating AE Package.");
        return;
    }

    QTextStream *modWriteStream = new QTextStream (modInfo);
    *modWriteStream << "AEInstallVersion -> "+aeVersion+"\n";
    *modWriteStream << "NameOfMod -> "+modName+"\n";
    *modWriteStream << "ModVersion -> "+version+"\n";
    *modWriteStream << "Creator -> "+authors+"\n";
    *modWriteStream << "Readme -> "+description.replace("\n"," \\n ")+"\n";
    if(!incompatiblePackages.isEmpty()){
        *modWriteStream << "IncompatibleWith -> "+incompatiblePackages+"\n";
    }
    if(!dependentPackages.isEmpty()){
        *modWriteStream << "DependsOn -> "+dependentPackages+"\n";
    }

    if(bslExist){
        if(bslReplace){
            *modWriteStream << "HasBsl -> Yes\n";
        }
        else{
            *modWriteStream << "HasBsl -> Addon\n";
        }
    }

    if(!unlockLevels.isEmpty()){
        *modWriteStream << "UnlockLevel -> "+unlockLevels+"\n";
    }

    *modWriteStream << "Vago -> "+GlobalVars::AppVersion;

    delete modWriteStream; //it auto closes the files/streams
    delete modInfo;

    //Create zipped package using PKZIP 2.0 (http://osdab.42cows.org/snippets/zip.php?mode=advanced)
    if(createZip){
        Zip uz;

        Zip::ErrorCode ec = uz.createArchive(this->packagesLocation+"/"+packageName+".zip");
        checkForZipError(ec);

        ec=uz.addDirectory(modDir);
        checkForZipError(ec);

        ec = uz.closeArchive();
        checkForZipError(ec);
    }

    if(openFolder){
        QDesktopServices::openUrl(QUrl("file:///"+this->packagesLocation));
    }
}

void PackageWizard::copyPackageFolders(const DropTableWidget *myTable, QString tableDir, QString modDir, bool &bslExist){

    QString sourceFolder;
    bool onisExist=false;
    QString path;

    for(int i=0; i<myTable->rowCount(); i++){

        sourceFolder=myTable->item(i,2)->text();

        if(myTable->item(i,1)->text()==".oni"){
            path=modDir+"oni/"+tableDir;
            if(!onisExist){
                onisExist=true;
            }
        }
        else if(myTable->item(i,1)->text()==".bsl"){
            path=modDir+"bsl/"+tableDir;
            if(!bslExist){
                bslExist=true;
            }
        }
        else{
            path=modDir+"patches/"+tableDir;
            if(!bslExist){
                bslExist=true;
            }
        }
        QDir().mkpath(path); //create path if doesn't exist
        if(!Util::cpDir(sourceFolder,path+Util::cutName(sourceFolder))){//copy contents (creates dest destination automatically if not exists yet)
            QString errorString="An error occurred while copping the folder/files to the package folder: \n"
                    "Copying from "+sourceFolder+"\n to "+path+Util::cutName(sourceFolder);
            Util::showErrorLogPopUp(errorString);
            this->myLogger->writeString(errorString);
        }
    }
}

/**
  Convenience function for checking for zipping errors
  */
void PackageWizard::checkForZipError(Zip::ErrorCode ec){
    if (ec != Zip::Ok){
        const QString error="Error found while zipping the package. Error number = "+QString::number(ec);
        Util::showErrorPopUp(error);
        this->myLogger->writeString(error);
    }
}
