#include "mainwindow.h"
#include "ui_mainwindow.h"

const QString MainWindow::VagoSettingsName = "settingsVago.ini";

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    this->AppDir=getOSIndependentAppPath();

    this->setWindowTitle("Vago v"+GlobalVars::AppVersion);

    if(!QFile::exists(this->AppDir+"/"+GlobalVars::OniSplitString)){
        Util::showErrorPopUp("OniSplit not found. Please download it at "+GlobalVars::ModsDomain+" and put it in the same folder of Vago. \n\nProgram will now exit.");
        exit(1);
    }

    if(!QFile::exists(this->AppDir+"/"+GlobalVars::XmlToolsString)){
        Util::showErrorPopUp("xmlTools not found. Please download it at "+GlobalVars::ModsDomain+" and put it in the same folder of Vago. \n\nProgram will now exit.");
        exit(1);
    }

    this->vagoSettings = new QSettings(this->AppDir + "/" + this->VagoSettingsName, QSettings::IniFormat);

    //First Execution? Old configuration? Settings missed?
    bool iniChanged=false;
    if(!this->vagoSettings->contains("VagoVersion") || this->vagoSettings->value("VagoVersion")!=GlobalVars::AppVersion){
        this->vagoSettings->setValue("VagoVersion", GlobalVars::AppVersion);
        iniChanged=true;
    }
    if(!this->vagoSettings->contains("Workspace")){
        this->vagoSettings->setValue("Workspace", this->AppDir+"/VagoWorkspace");
        iniChanged=true;
    }
    if(!this->vagoSettings->contains("AeFolder")){

        Util::showPopUp("Seems it's the first time you are executing Vago. \n\nPlease input your Anniversary Edition (AE) Folder.");
        QString aefolder=Util::normalizePath(QFileDialog::getExistingDirectory(this,"Choose Anniversary Edition (AE) folder..."));

        if(aefolder.isEmpty()){
            Util::showErrorPopUp("AE folder is mandatory. Application will now exit.");
            exit(1);
        }

        if(!aefolder.endsWith("AE")){
            Util::showWarningPopUp("Seems the folder you selected isn't called 'AE'. \n\nIf you run in any problems you can always change it in Vago preferences window.");
        }

        this->vagoSettings->setValue("AeFolder", aefolder);
        iniChanged=true;
    }
    if(!this->vagoSettings->contains("OniWindow")){
        this->vagoSettings->setValue("OniWindow", true);
        iniChanged=true;
    }
    if(!this->vagoSettings->contains("SeparateInWorkspace")){
        this->vagoSettings->setValue("SeparateInWorkspace",true);
        iniChanged=true;
    }
    if(!this->vagoSettings->contains("ConfirmExit")){
        this->vagoSettings->setValue("ConfirmExit", false);
        iniChanged=true;
    }

    if(iniChanged){
        this->vagoSettings->sync();
    }
    ///

    this->workspaceLocation=this->vagoSettings->value("Workspace").toString();
    this->workspaceWizardsLocation=this->workspaceLocation+"/Wizards";
    this->AeLocation=this->vagoSettings->value("AeFolder").toString();
    this->outputFolder=this->workspaceLocation;

    //Create our workspace if it doesn't exists yet
    if(!QDir(this->workspaceLocation).exists()){
        QDir().mkdir(this->workspaceLocation);
    }
    this->itemsLoaded=new QLabel(this);
    ui->statusBar->addWidget(this->itemsLoaded);
    this->myBar = new QProgressBar(this);
    this->myBar->setTextVisible(false); //hides text

    this->myBar->setSizePolicy(QSizePolicy::Minimum,QSizePolicy::Fixed);
    this->myBar->setMinimumWidth(150);
    this->myBar->hide(); //hide while not being used

    ui->statusBar->addPermanentWidget(myBar); //this adds automatically in right

    this->myLogger = new Logger(); //start logger

    //Initialize list pointers
    this->listToProccess = new QStringList;

    //Create a thread for do the conversion in background
    this->myConverter = new Converter(this->myLogger,this->listToProccess);

    // User interface
    ui->mainToolBar->addWidget(ui->tbAE); //add ae installer launch button
    ui->mainToolBar->addWidget(ui->emptySpacerLabel); //trick, we can't add directly a space so we add an empty
    ui->mainToolBar->addWidget(ui->tbOni); //add oni launch buttonlabel
    ui->mainToolBar->addWidget(ui->emptySpacerLabel2); //same as before
    ui->mainToolBar->addWidget(ui->tbCommand); //add option to manual onisplit commands
    ui->mainToolBar->addWidget(ui->emptySpacerLabel3); //same as before
    ui->mainToolBar->addWidget(ui->tbOpenFolder); //add option to open folder with files converted etc

    ui->mainToolBar->setLayoutDirection(Qt::RightToLeft);

    setConverterButtonsSize();

#ifdef Q_WS_MAC
    // setUnifiedTitleAndToolBarOnMac(true); // Qt suggests to use it on mac | http://www.slideshare.net/qtbynokia/how-to-make-your-qt-app-look-native // align on left doesn't work if active
    ui->tbOni->setIcon(QIcon(":/new/icons/oni_icon_mac.png")); // Oni executable on mac have a different icon than windows
    // Set mac platform the first one in the menu, and also make it checkable by default
    ui->menuTarget_Platform->removeAction(ui->actionWindows);
    ui->menuTarget_Platform->addAction(ui->actionWindows);
    ui->actionWindows->setChecked(false);
    ui->actionMac_Windows_demo->setChecked(true);
    resize(800,600); // Mac OS pcs should be able to render this resolution without any problem. It's also better
    // because the components on mac use more space
#else
    resize(640,480);
#endif

    connectSlots();

    //Commands Mapping
    this->commandMap = QHash<QString, QString>();
    mapCommands();

    updateItemsLoaded(ui->twSourcesGeneral);

    this->myLogger->writeString("Application started.");
}

MainWindow::~MainWindow()
{
    delete ui;
    this->myLogger->writeString("Application Exited.");
}

void MainWindow::on_actionExit_triggered()
{
    close();
}

void MainWindow::on_actionAbout_triggered()
{
    //Show preferences
    About *aboutWindow = new About(this);
    aboutWindow->show(); //it destroys itself when finished.
}

void MainWindow::on_actionAE_Package_Creator_triggered()
{
    PackageWizard myWizard = PackageWizard(this->workspaceWizardsLocation, this->vagoSettings, this->myLogger);
    myWizard.exec();
}

void MainWindow::on_actionSound_Wizard_triggered()
{
    SoundWizard myWizard (this->AppDir, this->workspaceWizardsLocation, this->myLogger, &this->commandMap);
    myWizard.exec();
}

void MainWindow::on_tbOni_clicked()
{
    QStringList arguments;

    if(this->vagoSettings->value("OniWindow").toBool()){ // Run in a window?
        arguments << "-noswitch";
    }
#ifdef Q_WS_WIN
    else{
        arguments << "-switch"; // only supported on windows. Was added by daodan dll.
    }
#endif

    arguments << "-debugfiles";

    if(!QProcess::startDetached(this->AeLocation+"/"+GlobalVars::OniExe,arguments,this->AeLocation)){
        showErrStatusMessage("Oni could not be started!");
    }
}

void MainWindow::on_tbAE_clicked()
{
    // If the app turn out someday to a native app use QProcess::startDetached instead...

    if(!QDesktopServices::openUrl("file:///"+this->AeLocation+"/AEInstaller/bin/AEInstaller2.jar")){
        showErrStatusMessage("Could not start AE Installer!");
    }
}

void MainWindow::on_tbOpenFolder_clicked()
{
    QDesktopServices::openUrl(QUrl("file:///"+this->outputFolder));
}

void MainWindow::on_cbEnvMap_toggled(bool checked)
{
    ui->leEnvMapTexture->setEnabled(checked);
}

void MainWindow::on_cbTexture_toggled(bool checked)
{
    ui->leTextureName->setEnabled(checked);
}

void MainWindow::on_cbWithAnimation_toggled(bool checked)
{
    ui->leAnimationName->setEnabled(checked);
}

void MainWindow::on_cbCamera_toggled(bool checked)
{
    if(checked){
        ui->cbGeometry->setChecked(false);
    }
}

void MainWindow::on_cbGeometry_toggled(bool checked)
{
    ui->leGeometryName->setEnabled(checked);
    if(checked){
        ui->cbCamera->setChecked(false);
    }
}

void MainWindow::on_actionCheck_For_Updates_triggered()
{

    //let's check in the web if this version is the latest
    QNetworkAccessManager *manager = new QNetworkAccessManager(this);
    connect(manager, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(checkVagoLastVersion(QNetworkReply*)));

    manager->get(QNetworkRequest(QUrl(GlobalVars::VagoCheckUpdatesUrl)));

}

void MainWindow::checkVagoLastVersion(QNetworkReply *result){

    if(result->error()==QNetworkReply::NoError){
        QScriptEngine engine;
        QScriptValue sc = engine.evaluate("(" + QString(result->readAll()) + ")");

        // "field_version":{"und":[{"value":"0.6a","format":null,"safe_value":"0.6a"}]} //Note the use of [{}] which means it's a array of 1 element with one object inside (so the use of [0] bellow

        QString newVersion = sc.property("field_version").toObject().property("und").toObject().property("0").toObject().property("value").toString();

        if(newVersion!=GlobalVars::AppVersion){
            Util::showRichPopUp("There's a new version of Vago! (v"+newVersion+")<br/><br/>"+
                                "You can download it <a href='"+GlobalVars::VagoWebUrl+"'>here</a>.");
        }
        else{
            Util::showPopUp("You are using last version.");
        }
    }
    else{
        Util::showErrorPopUp("An error occurred checking last version:\n\n"+result->errorString());
    }
    result->deleteLater();
}

void MainWindow::on_pbAddSourceGeneral_clicked()
{
    if(QString::compare(ui->cbFromGeneral->currentText(),"ONI",Qt::CaseSensitive)==0 && QString::compare(ui->cbToGeneral->currentText(),"DAT",Qt::CaseSensitive)==0){ //CaseSensitive is faster)
        addFilesSource(ui->twSourcesGeneral,Util::multipleDirDialog("Choose folders with ONIs..."));
    }
    else{
        addFilesSource( ui->twSourcesGeneral,QFileDialog::getOpenFileNames(this,"Choose the files...","./" , "All Files (*.*)"));
    }
}

void MainWindow::on_pbAddSourceTextures_clicked()
{
    addFilesSource( ui->twSourcesTextures, QFileDialog::getOpenFileNames(this,"Choose the files...","./" , "All Files (*.*)"));
}

void MainWindow::on_pbAddSourceModels_clicked()
{
    addFilesSource( ui->twSourcesModels,QFileDialog::getOpenFileNames(this,"Choose the files...","./" , "All Files (*.*)"));
}

void MainWindow::on_pbAddSourceAnimations_clicked()
{
    addFilesSource( ui->twSourcesAnimations,QFileDialog::getOpenFileNames(this,"Choose the files...","./" , "All Files (*.*)"));
}

void MainWindow::on_pbAddSourceLevels_clicked()
{
    addFilesSource( ui->twSourcesLevels,QFileDialog::getOpenFileNames(this,"Choose the files...","./" , "All Files (*.*)"));
}

void MainWindow::on_pbAddSourceMisc_clicked()
{
    addFilesSource( ui->twSourcesMisc,QFileDialog::getOpenFileNames(this,"Choose the files...","./" , "All Files (*.*)"));
}

QString MainWindow::getFileOutputFolder(QString fromTo, QString myOutputFolder){

    if(myOutputFolder==""){ //We may want to change to a non standart location with context menu
        myOutputFolder=this->outputFolder;
    }

    if(this->vagoSettings->value("SeparateInWorkspace").toBool() && myOutputFolder==this->workspaceLocation){
        myOutputFolder+="/"+ui->tabWidget->tabText(ui->tabWidget->currentIndex());
        myOutputFolder+="/"+QString(fromTo).replace(" / ","_").replace(" > "," - ");
    }
    return Util::insertQuotes(myOutputFolder+"/");
}

void MainWindow::addFilesSource(DropTableWidget *myTable, QStringList files){

    //Get Conversion pretended
    QString from,to;

    QString fromTo = getTypeConversion(myTable);

    from = QString(fromTo).remove(fromTo.indexOf(" >"),fromTo.size()-1); //parse the string to get the from, only 1 time parsed by each group of files = very fast
    to = QString(fromTo).remove(0,fromTo.lastIndexOf("> ")+2); //+2 to start after "> "

    //Pre-processing (check if the files/folders received are valid), e.g. check for ONI->DAT if are only given folders and not files
    if(QString::compare(from,"ONI",Qt::CaseSensitive)==0 && QString::compare(to,"DAT",Qt::CaseSensitive)==0){
        //check if it's a folder
        foreach(QString myFile, files){
            if(!QDir(myFile).exists()){
                showErrStatusMessage("Only folders are allowed for this operation.");
                return;
            }
        }

    }
    else{
        foreach(QString myFile, files){
            //check if it's a file
            if(QDir(myFile).exists()){
                showErrStatusMessage("Only files are allowed for this operation.");
                return;
            }
        }
    }

    //Build command
    QString command, lastFileName;

    QString myOutputFolder=getFileOutputFolder(fromTo);

    //if folder doesn't exist onisplit will create it for us :)
    foreach(QString currentFile, files){

        currentFile=Util::normalizeAndQuote(currentFile); //insert quotes ("") in file

        if(lastFileName.isEmpty()){ //Optimization: all commands are the same for each file, just replace the filename

            command=getCommand(myTable,myOutputFolder,from,to,currentFile);

            if(command.isEmpty()){ //something wrong was happening (not inputted a texture name?)
                return; //stop adding files
            }
            currentFile=Util::cutName(currentFile);
        }else{ //one parsing was already made just replace the filename by the old one in the command

            currentFile=Util::cutName(currentFile);

            command.replace(lastFileName,currentFile,Qt::CaseSensitive); //case sentive is faster
        }

        lastFileName=currentFile;

        addRowTable(myTable,lastFileName,fromTo,command);
    }
    updateItemsLoaded(myTable);
}

QString MainWindow::fileParsingGeneral(QString myOutputFolder, QString from, QString to , QString file){

    QString command;

    if(QString::compare(from,"ONI",Qt::CaseSensitive)==0 && QString::compare(to,"DAT",Qt::CaseSensitive)==0){ //CaseSensitive is faster

        QString datName;

        if(ui->cbDatGeneral->isChecked()){
            if(ui->leTargetDatGeneral->text().isEmpty()){
                showErrStatusMessage("Checkbox '"+ui->cbDatGeneral->text()+"' is selected. The name cannot be empty.");
                return "";
            }
            datName+=QString(myOutputFolder).insert(myOutputFolder.size()-1,ui->leTargetDatGeneral->text()); //set name inputted by user
            if(!ui->leTargetDatGeneral->text().toUpper().endsWith(".DAT")){
                datName.insert(datName.size()-1,".dat"); //append extension if necessary (-1 to maintain final quote)
            }
        }
        else{
            datName=QString(myOutputFolder).insert(myOutputFolder.size()-1,Util::cutName(file).remove("/")+".dat"); //if none iputted set the same name of input file
        }

        if(ui->actionWindows->isChecked()){ //is target plataform select windows?
            return command=this->commandMap.value("general->"+from+"->"+to+"(PC)")+" "+ file + " "+datName;
        }
        else{
            return command=this->commandMap.value("general->"+from+"->"+to+"(demoPCMAC)")+" "+ file + " "+datName;
        }
    }
    else if(QString::compare(from,"ONI",Qt::CaseSensitive)==0 && QString::compare(to,"XML",Qt::CaseSensitive)==0 && ui->cbTRAMGeneral->isChecked()){
        if(ui->leTRAMGeneral->text().isEmpty()){
            showErrStatusMessage("Checkbox '"+ui->cbTRAMGeneral->text()+"' is selected. The source cannot be empty.");
            return "";
        }
        return command=this->commandMap.value("general->"+from+"->"+to)+" "+myOutputFolder+" "+this->commandMap.value("general->"+ui->cbTRAMGeneral->text())+file + " "+ Util::normalizeAndQuote(ui->leTRAMGeneral->text());
    }
    else{
        return command=this->commandMap.value("general->"+from+"->"+to)+" "+myOutputFolder+" "+file;
    }

}

QString MainWindow::fileParsingTextures(QString myOutputFolder, QString from, QString to , QString file){

    QString command=this->commandMap.value("textures->"+from+"->"+to)+" "+myOutputFolder;

    if(ui->gbTextures->isEnabled()){ //faster than compare strings (if is DAT/ONI)

        if(ui->cbMipMapsTextures->isChecked()){
            command+=" "+this->commandMap.value("textures->"+ui->cbMipMapsTextures->text());
        }

        if(ui->cbNoUwrap->isChecked()){
            command+=" "+this->commandMap.value("textures->"+ui->cbNoUwrap->text());
        }

        if(ui->cbNoVwrap->isChecked()){
            command+=" "+this->commandMap.value("textures->"+ui->cbNoVwrap->text());
        }

        if(ui->cbLarge->isChecked()){
            command+=" "+this->commandMap.value("textures->"+ui->cbLarge->text());
        }

        if(ui->rbBGR32->isChecked()){
            command+=" "+this->commandMap.value("textures->"+ui->rbBGR32->text());
        }
        else if(ui->rbBGRA32->isChecked()){
            command+=" "+this->commandMap.value("textures->"+ui->rbBGRA32->text());
        }
        else if(ui->rbBGR555->isChecked()){
            command+=" "+this->commandMap.value("textures->"+ui->rbBGR555->text());
        }
        else if(ui->rbBGRA5551->isChecked()){
            command+=" "+this->commandMap.value("textures->"+ui->rbBGRA5551->text());
        }
        else if(ui->rbBGRA444->isChecked()){
            command+=" "+this->commandMap.value("textures->"+ui->rbBGRA444->text());
        }
        else{ //dxt1 checked
            command+=" "+this->commandMap.value("textures->"+ui->rbDxt1->text());
        }

        if(ui->cbEnvMap->isChecked()){
            if(ui->leEnvMapTexture->text().isEmpty()){
                showErrStatusMessage("Checkbox '"+ui->cbEnvMap->text()+"' is selected. The name texture name cannot be empty.");
                return "";
            }
            command+=" "+this->commandMap.value("textures->"+ui->cbEnvMap->text()) + ui->leEnvMapTexture->text().remove(".oni",Qt::CaseInsensitive);
        }
    }

    return command+=" "+file; //add source
}

QString MainWindow::fileParsingModels(QString myOutputFolder, QString from, QString to , QString file){

    QString command=this->commandMap.value("models->"+from+"->"+to)+" "+myOutputFolder;

    //TODO: This can be optimized. When some are not enable others are.
    if(ui->cbTexture->isChecked()){
        if(ui->leTextureName->text().isEmpty()){
            showErrStatusMessage("Checkbox '"+ui->cbTexture->text()+"' is selected. The name cannot be empty.");
            return "";
        }
        command+=" "+this->commandMap.value("models->"+ui->cbTexture->text()) + ui->leTextureName->text().remove(".oni",Qt::CaseInsensitive);
    }

    if(ui->cbCellShading->isChecked()){
        command+=" "+this->commandMap.value("models->"+ui->cbCellShading->text());
    }

    if(ui->cbNormals->isChecked()){
        command+=" "+this->commandMap.value("models->"+ui->cbNormals->text());
    }

    if(ui->cbWithAnimation->isEnabled()){
        if(ui->cbWithAnimation->isChecked()){
            command+=" "+this->commandMap.value("models->"+ui->cbWithAnimation->text())+ui->leAnimationName->text().remove(".oni",Qt::CaseInsensitive);
        }
        else{
            command+=" "+this->commandMap.value("models->No Animation");
        }
    }


    return command+=" "+file; //add source
}

QString MainWindow::fileParsingAnimations(QString myOutputFolder, QString from, QString to , QString file){

    QString command=this->commandMap.value("animations->"+from+"->"+to)+" "+myOutputFolder + " " + file ;

    if(ui->cbCamera->isChecked()){
        command+=" "+this->commandMap.value("animations->"+ui->cbCamera->text());
    }
    else if(ui->cbGeometry->isChecked()){
        if(ui->leGeometryName->text().isEmpty()){
            showErrStatusMessage("Checkbox '"+ui->cbGeometry->text()+"' is selected. The geometry file path cannot be empty.");
            return "";
        }
        command+=" "+this->commandMap.value("animations->"+ui->cbGeometry->text()) + (ui->leGeometryName->text().startsWith('"')?ui->leGeometryName->text():Util::insertQuotes(ui->leGeometryName->text()));
    }

    return command;
}

QString MainWindow::fileParsingLevels(QString myOutputFolder, QString from, QString to , QString file){

    QString datName, command;

    command=this->commandMap.value("levels->"+from+"->"+to)+" "+myOutputFolder+" "+file;

    if(from=="MASTER XML" && to=="DAT"){
        command+=GlobalVars::OniSplitProcSeparator; //insert mark so we know this action will take 2 commands

        QString datName;
        if(ui->cbDatLevels->isChecked()){
            if(ui->leTargetDatLevels->text().isEmpty()){
                showErrStatusMessage("Checkbox '"+ui->cbDatLevels->text()+"' is selected. The name cannot be empty.");
                return "";
            }
            datName+=QString(myOutputFolder).insert(myOutputFolder.size()-1,ui->leTargetDatLevels->text()); //set name inputted by user
            if(!ui->leTargetDatLevels->text().toUpper().endsWith(".DAT")){
                datName.insert(datName.size()-1,".dat"); //append extension if necessary (-1 to maintain final quote)
            }
        }
        else{
            datName=QString(myOutputFolder).insert(myOutputFolder.size()-1,Util::cutName(file).remove("/").replace(".xml",".dat",Qt::CaseInsensitive)); //if none iputted set the same name of input file
        }

        if(ui->actionWindows->isChecked()){ //is target plataform select windows?
            command+=this->commandMap.value("general->ONI->"+to+"(PC)")+" "+myOutputFolder+" "+datName; //add second command
        }
        else{
            command+=this->commandMap.value("general->ONI->"+to+"(demoPCMAC)")+" "+myOutputFolder+" "+datName; //add second command
        }
    }

    if(ui->cbBnvLevels->isChecked()){

        if(ui->leBnvLevels->text().isEmpty()){
            showErrStatusMessage("Checkbox '"+ui->cbBnvLevels->text()+"' is selected. The BNV file cannot be empty.");
            return "";
        }
        command+=" "+Util::normalizeAndQuote(ui->leBnvLevels->text());
    }

    if(ui->cbAdditionalSourcesLevels->isChecked()){

        if(ui->leAdditSourcesLevels->text().isEmpty()){
            showErrStatusMessage("Checkbox '"+ui->cbAdditionalSourcesLevels->text()+"' is selected. The source files cannot be empty.");
            return "";
        }

        QString additionalFiles=ui->leAdditSourcesLevels->text();

        int currentIndex=0, nextIndex=0;

        //parse all files (separated by spaces)
        while(true){
            nextIndex=additionalFiles.indexOf(" ",currentIndex+1);

            command += " "+Util::normalizeAndQuote(additionalFiles.mid(currentIndex,(nextIndex-currentIndex)));

            if(nextIndex==-1){ //we got to the end, stop parsing
                break;
            }
            currentIndex=nextIndex+1; //update currentIndex +1 for start after the separator
        }
    }

    if(ui->cbGridsLevels->isChecked()){
        command+=GlobalVars::OniSplitProcSeparator+this->commandMap.value("levels->"+ui->cbGridsLevels->text())+" "+Util::normalizeAndQuote(ui->leBnvLevels->text())+" "+file+" -out:"+myOutputFolder;
    }

    return command;
}

QString MainWindow::fileParsingMisc(QString myOutputFolder, QString from, QString to , QString file){
    return this->commandMap.value("misc->"+from+"->"+to)+" "+myOutputFolder+" "+file;
}

void MainWindow::addRowTable(DropTableWidget *myTable, QString fileName, QString fromTo, QString command){
    //Get actual number rows
    int twSize=myTable->rowCount();

    //increase the rows for the new item
    myTable->setRowCount(twSize+1);

    //Add to table and list to
    QTableWidgetItem *newFile = new QTableWidgetItem(fileName);
    QTableWidgetItem *newConversion = new QTableWidgetItem(fromTo);
    QTableWidgetItem *newCommand = new QTableWidgetItem(command);

    myTable->setItem(twSize,0,newFile);
    myTable->setItem(twSize,1,newConversion);
    myTable->setItem(twSize,2,newCommand);

    myTable->updateTableToolTips(twSize); //Update tool tips
}

void MainWindow::on_pbConvertGeneral_clicked()
{
    startConversion(ui->twSourcesGeneral);
}

void MainWindow::on_pbConvertTextures_clicked()
{
    startConversion(ui->twSourcesTextures);
}

void MainWindow::on_pbConvertModels_clicked()
{
    startConversion(ui->twSourcesModels);
}

void MainWindow::on_pbConvertAnimations_clicked()
{
    startConversion(ui->twSourcesAnimations);
}

void MainWindow::on_pbConvertLevels_clicked()
{
    startConversion(ui->twSourcesLevels);
}

void MainWindow::on_pbConvertMisc_clicked()
{
    startConversion(ui->twSourcesMisc);
}

void MainWindow::startConversion(DropTableWidget *myTable){

    bool ready=false;
    for(int i=0; i<myTable->rowCount(); i++){ //There are items to process?
        if(myTable->item(i,2)->background()!=myTable->disabledBackStyle){
            ready=true;
            break;
        }
    }

    if(!ready){
        showErrStatusMessage("Please add sources to convert first.");
        return;
    }

    if(myBar->isVisible()){
        Util::showErrorPopUp("Another conversion is progress. Please wait until it finishes.");
        return;
    }

    for(int i=0; i<myTable->rowCount(); i++){
        //Only process enabled items
        if(myTable->item(i,2)->background()!=myTable->disabledBackStyle){
            this->listToProccess->append(myTable->item(i,2)->text());
        }
    }

    this->myConverter->start();
}

void MainWindow::TsetupProgressBar(int max){
    this->myBar->setValue(0);
    this->myBar->show();
    this->myBar->setMaximum(max);
}

void  MainWindow::TupdateProgressBar(){
    this->myBar->setValue(this->myBar->value()+1); //more one task done
}

void MainWindow::TresultConversion(QString result, int numErrors){
    QApplication::alert(this); //Show a notification if window is not active
    this->myBar->hide();

    if(numErrors!=0){
        QString sNumErrors=QString::number(numErrors);
        if(numErrors>1){
            Util::showErrorLogPopUp(result+"\n This is the last of "+sNumErrors+" Errors.");
            showErrStatusMessage("Something gone wrong. Check log file ("+sNumErrors+" Errors).");
        }
        else{
            Util::showErrorLogPopUp(result);
            showErrStatusMessage("Something gone wrong. Check log file.");
        }

    }
    else{
        showSuccessStatusMessage("Everything went well!");
    }
}

void MainWindow::showErrStatusMessage(QString message){

    QPalette myPalete = QPalette();
    myPalete.setColor( QPalette::WindowText, QColor(255,0,0));
    statusBar()->setPalette( myPalete );
    ui->statusBar->showMessage(message,10000); //display by 10 seconds

}

void MainWindow::showSuccessStatusMessage(QString message){

    QPalette myPalete = QPalette();
    myPalete.setColor( QPalette::WindowText, QColor(0,150,0));
    statusBar()->setPalette( myPalete );
    ui->statusBar->showMessage(message,10000); //display by 10 seconds

}

void MainWindow::mapCommands(){
    ////////////////////////////////////////////////////////////////////////General Commands
    this->commandMap.insert("general->DAT->ONI","-export");
    //this->commandMap.insert("general->ONI->DAT","-import"); //Not used.
    this->commandMap.insert("general->ONI->DAT(PC)","-import:nosep");
    this->commandMap.insert("general->ONI->DAT(demoPCMAC)","-import:sep");
    this->commandMap.insert("general->ONI->XML","-extract:xml");
    this->commandMap.insert("general->XML->ONI","-create");
    //######################General Options
    this->commandMap.insert("general->"+ui->cbTRAMGeneral->text(),"-anim-body:");
    //Possible Combinations
    this->commandMap.insertMulti("general->DAT","ONI");
    this->commandMap.insertMulti("general->ONI","DAT");
    this->commandMap.insertMulti("general->ONI","XML");
    this->commandMap.insertMulti("general->XML","ONI");

    ////////////////////////////////////////////////////////////////////////Textures Commands
    this->commandMap.insert("textures->DAT / ONI->DDS","-extract:dds");
    this->commandMap.insert("textures->DAT / ONI->TGA","-extract:tga");
    this->commandMap.insert("textures->DAT / ONI->PNG","-extract:png");
    this->commandMap.insert("textures->DAT / ONI->JPG","-extract:jpg");
    this->commandMap.insert("textures->DDS / TGA / PNG / JPG->ONI","-create:txmp");
    //######################Textures Options
    this->commandMap.insert("textures->"+ui->rbBGR32->text(),"-format:bgr32");
    this->commandMap.insert("textures->"+ui->rbBGRA32->text(),"-format:bgra32");
    this->commandMap.insert("textures->"+ui->rbBGR555->text(),"-format:bgr555");
    this->commandMap.insert("textures->"+ui->rbBGRA5551->text(),"-format:bgra5551");
    this->commandMap.insert("textures->"+ui->rbBGRA444->text(),"-format:bgra4444");
    this->commandMap.insert("textures->"+ui->rbDxt1->text(),"-format:dxt1");
    this->commandMap.insert("textures->"+ui->cbMipMapsTextures->text(),"-genmipmaps");
    this->commandMap.insert("textures->"+ui->cbNoUwrap->text(),"-nouwrap");
    this->commandMap.insert("textures->"+ui->cbNoVwrap->text(),"-novwrap");
    this->commandMap.insert("textures->"+ui->cbLarge->text(),"-large");
    this->commandMap.insert("textures->"+ui->cbEnvMap->text(),"-envmap:");
    //Possible Combinations
    this->commandMap.insertMulti("textures->DAT / ONI","DDS");
    this->commandMap.insertMulti("textures->DAT / ONI","TGA");
    this->commandMap.insertMulti("textures->DAT / ONI","PNG");
    this->commandMap.insertMulti("textures->DAT / ONI","JPG");
    this->commandMap.insertMulti("textures->DDS / TGA / PNG / JPG","ONI");

    ////////////////////////////////////////////////////////////////////////Models Commands
    this->commandMap.insert("models->ONI->OBJ","-extract:obj");
    this->commandMap.insert("models->ONI->DAE","-extract:dae -search "+Util::insertQuotes(this->AeLocation+"/GameDataFolder/level0_Final"));
    this->commandMap.insert("models->OBJ->ONI","-create:m3gm");
    this->commandMap.insert("models->DAE->ONI","-create:trbs");
    //######################Models Options
    this->commandMap.insert("models->"+ui->cbCellShading->text(),"-cel");
    this->commandMap.insert("models->"+ui->cbNormals->text(),"-normals");
    this->commandMap.insert("models->"+ui->cbTexture->text(),"-tex:");
    this->commandMap.insert("models->"+ui->cbWithAnimation->text(),"-anim:");
    this->commandMap.insert("models->No Animation","-noanim"); //No label with this name so can't be dynamic
    //Possible Combinations
    this->commandMap.insertMulti("models->ONI","OBJ");
    this->commandMap.insertMulti("models->ONI","DAE");
    this->commandMap.insertMulti("models->OBJ","ONI");
    this->commandMap.insertMulti("models->DAE","ONI");

    ////////////////////////////////////////////////////////////////////////Animations Commands
    this->commandMap.insert("animations->ONI->DAE","-extract:dae");
    this->commandMap.insert("animations->FILM DAT->XML","film2xml");
    //######################Animations Options
    this->commandMap.insert("animations->"+ui->cbCamera->text(),"-geom:camera");
    this->commandMap.insert("animations->"+ui->cbGeometry->text(),"-geom:");
    //Possible Combinations
    this->commandMap.insertMulti("animations->ONI","DAE");
    this->commandMap.insertMulti("animations->DAE","ONI");
    this->commandMap.insertMulti("animations->FILM DAT","XML");

    ////////////////////////////////////////////////////////////////////////Levels Commands
    this->commandMap.insert("levels->ONI->DAE","-extract:dae -search "+Util::insertQuotes(this->AeLocation+"/GameDataFolder/level0_Final"));
    this->commandMap.insert("levels->DAE->ONI","-create:akev");
    this->commandMap.insert("levels->MASTER XML->DAT","-create:level");
    this->commandMap.insert("levels->MASTER XML->ONI FILES","-create:level");
    //######################Levels Options
    this->commandMap.insert("levels->"+ui->cbGridsLevels->text(),"-grid:create");
    //Possible Combinations
    this->commandMap.insertMulti("levels->ONI","DAE");
    this->commandMap.insertMulti("levels->DAE","ONI");
    this->commandMap.insertMulti("levels->MASTER XML","DAT");
    this->commandMap.insertMulti("levels->MASTER XML","ONI FILES");

    ////////////////////////////////////////////////////////////////////////Misc Commands
    this->commandMap.insert("misc->DAT / ONI->WAV","-extract:wav");
    this->commandMap.insert("misc->DAT / ONI->AIF","-extract:aif");
    this->commandMap.insert("misc->DAT / ONI->TXT","-extract:txt");
    this->commandMap.insert("misc->WAV / AIF->ONI","-create");
    this->commandMap.insert("misc->TXT->ONI","-create:subt");
    //Possible Combinations
    this->commandMap.insertMulti("misc->DAT / ONI","WAV");
    this->commandMap.insertMulti("misc->DAT / ONI","AIF");
    this->commandMap.insertMulti("misc->DAT / ONI","TXT");
    this->commandMap.insertMulti("misc->WAV / AIF","ONI");
    this->commandMap.insertMulti("misc->TXT","ONI");

}

void MainWindow::on_cbFromGeneral_currentIndexChanged(const QString &arg1)
{
    updateComboBox(arg1, ui->cbToGeneral, "general");
}

void MainWindow::on_cbFromTextures_currentIndexChanged(const QString &arg1)
{
    //Options are only used for DAT/ONI -> Image
    if(QString::compare(arg1,"DAT / ONI",Qt::CaseSensitive)==0){ //case sensitive is faster
        ui->gbTextures->setEnabled(false);
    }
    else{
        ui->gbTextures->setEnabled(true);
    }

    updateComboBox(arg1, ui->cbToTextures, "textures");
}

void MainWindow::on_cbFromModels_currentIndexChanged(const QString &arg1)
{

    ui->cbCellShading->setEnabled(false);
    ui->cbCellShading->setChecked(false);
    ui->cbNormals->setEnabled(false);
    ui->cbNormals->setChecked(false);
    ui->cbTexture->setEnabled(false);
    ui->cbTexture->setChecked(false);
    ui->cbWithAnimation->setEnabled(false);
    ui->cbWithAnimation->setChecked(false);

    if(QString::compare(arg1,"OBJ",Qt::CaseSensitive)==0){ //case sensitive is faster
        ui->cbTexture->setEnabled(true);
    }
    else if(QString::compare(arg1,"DAE",Qt::CaseSensitive)==0){
        ui->cbCellShading->setEnabled(true);
        ui->cbNormals->setEnabled(true);
    }

    updateComboBox(arg1, ui->cbToModels, "models");
}

void MainWindow::on_cbFromAnimations_currentIndexChanged(const QString &arg1)
{
    ui->cbCamera->setEnabled(false);
    ui->cbCamera->setChecked(false);
    ui->cbGeometry->setEnabled(false);
    ui->cbGeometry->setChecked(false);

    if(QString::compare(arg1,"ONI",Qt::CaseSensitive)==0){ //case sensitive is faster
        ui->cbCamera->setEnabled(true);
        ui->cbGeometry->setEnabled(true);
    }

    updateComboBox(arg1, ui->cbToAnimations, "animations");
}

void MainWindow::on_cbFromLevels_currentIndexChanged(const QString &arg1)
{
    updateComboBox(arg1, ui->cbToLevels, "levels");
}

void MainWindow::on_cbFromMisc_currentIndexChanged(const QString &arg1)
{
    updateComboBox(arg1, ui->cbToMisc, "misc");
}

void MainWindow::updateComboBox(const QString &arg1, QComboBox *comboBox, const QString &identifier){
    comboBox->clear();

    QStringList toUpdate=QStringList();

    QStringList values=commandMap.values(identifier+"->"+arg1);

    for (int i = values.size()-1; i >= 0; i--){ //By defaut MultiItems have the inversed order (http://qt-project.org/doc/qt-4.8/qhash.html#insertMulti)
        toUpdate << values.at(i);
    }

    comboBox->addItems(toUpdate);
}


void MainWindow::on_actionWindows_triggered()
{
    ui->actionWindows->setChecked(true);
    ui->actionMac_Windows_demo->setChecked(false);
}

void MainWindow::on_actionMac_Windows_demo_triggered()
{
    ui->actionMac_Windows_demo->setChecked(true);
    ui->actionWindows->setChecked(false);
}

void MainWindow::on_pbRemoveSourceGeneral_clicked()
{
    removeTableContents( ui->twSourcesGeneral);
}

void MainWindow::on_pbRemoveSourceTextures_clicked()
{
    removeTableContents(ui->twSourcesTextures);
}

void MainWindow::on_pbRemoveSourceModels_clicked()
{
    removeTableContents(ui->twSourcesModels);
}

void MainWindow::on_pbRemoveSourceAnimations_clicked()
{
    removeTableContents(ui->twSourcesAnimations);
}

void MainWindow::on_pbRemoveSourceLevels_clicked()
{
    removeTableContents(ui->twSourcesLevels);
}

void MainWindow::on_pbRemoveSourceMisc_clicked()
{
    removeTableContents(ui->twSourcesMisc);
}

void MainWindow::on_pbClearSourcesGeneral_clicked()
{
    clearTableContents(ui->twSourcesGeneral);
}

void MainWindow::on_pbClearSourcesTextures_clicked()
{
    clearTableContents(ui->twSourcesTextures);
}

void MainWindow::on_pbClearSourcesModels_clicked()
{
    clearTableContents(ui->twSourcesModels);
}

void MainWindow::on_pbClearSourcesAnimations_clicked()
{
    clearTableContents(ui->twSourcesAnimations);
}

void MainWindow::on_pbClearSourcesLevels_clicked()
{
    clearTableContents(ui->twSourcesLevels);
}

void MainWindow::on_pbClearSourcesMisc_clicked()
{
    clearTableContents(ui->twSourcesMisc);
}

void MainWindow::removeTableContents(DropTableWidget *myTable){
    int size = myTable->selectionModel()->selectedRows().size();

    if(size==0){
        Util::showPopUp("Select a row first.");
        return;
    }

    if(Util::showQuestionPopUp(this,"Are you sure you want to delete the selected rows?")){
        for(int i=0; i<size; i++){
            //myTable->removeRow(myTable->selectedItems().at(size-i-1)->row());
            myTable->removeRow(myTable->selectionModel()->selectedRows().at(size-i-1).row());
        }
        updateItemsLoaded(myTable);
    }
}

void MainWindow::clearTableContents(DropTableWidget *myTable){
    if(myTable->rowCount()==0){
        Util::showPopUp("Nothing to clear.");
        return;
    }

    if(Util::showQuestionPopUp(this,"Are you sure you want to clear the content?")){
        myTable->clearContents();
        myTable->setRowCount(0);
    }
    updateItemsLoaded(myTable);
}


void MainWindow::on_actionPreferences_triggered()
{
    //Show preferences
    Preferences *preferencesWindow = new Preferences(this,this->vagoSettings);
    preferencesWindow->exec(); //it destroys itself when finished.
}


void MainWindow::closeEvent(QCloseEvent *event){
    if(this->vagoSettings->value("ConfirmExit").toBool()){
        if(!Util::showQuestionPopUp(this,"Exit Vago?")){
            event->ignore();
        }
    }
}

void MainWindow::on_cbToGeneral_currentIndexChanged(const QString &arg1)
{

    ui->cbDatGeneral->setEnabled(false);
    ui->cbDatGeneral->setChecked(false);
    ui->cbTRAMGeneral->setEnabled(false);
    ui->cbTRAMGeneral->setChecked(false);

    if(QString::compare(ui->cbFromGeneral->currentText(),"ONI",Qt::CaseSensitive)==0){
        if(QString::compare(arg1,"DAT",Qt::CaseSensitive)==0){
            ui->cbDatGeneral->setEnabled(true);
        }
        else{
            ui->cbTRAMGeneral->setEnabled(true);
        }
    }

}

void MainWindow::on_cbToModels_currentIndexChanged(const QString &arg1)
{
    ui->cbWithAnimation->setEnabled(false);
    ui->cbWithAnimation->setChecked(false);

    if(arg1=="DAE"){
        ui->cbWithAnimation->setEnabled(true);
    }
}

void MainWindow::on_cbToLevels_currentIndexChanged(const QString &arg1)
{

    ui->cbDatLevels->setEnabled(false);
    ui->cbDatLevels->setChecked(false);
    ui->cbBnvLevels->setEnabled(false);
    ui->cbBnvLevels->setChecked(false);
    ui->cbAdditionalSourcesLevels->setEnabled(false);
    ui->cbAdditionalSourcesLevels->setChecked(false);
    ui->cbGridsLevels->setEnabled(false);
    ui->cbGridsLevels->setChecked(false);

    if(ui->cbFromLevels->currentText()=="MASTER XML" && arg1=="DAT"){
        ui->cbDatLevels->setEnabled(true);
    }
    else if(ui->cbFromLevels->currentText()=="DAE" && arg1=="ONI"){
        ui->cbBnvLevels->setEnabled(true);
        ui->cbAdditionalSourcesLevels->setEnabled(true);
    }
}

void MainWindow::on_cbDatGeneral_toggled(bool checked)
{
    ui->leTargetDatGeneral->setEnabled(checked);
}

void MainWindow::on_cbTRAMGeneral_toggled(bool checked)
{
    ui->leTRAMGeneral->setEnabled(checked);
    if(checked){
        QString file=QFileDialog::getOpenFileName(this,"Choose the TRAM.oni file...","./" , "All Files (*.*)");
        if(!file.isEmpty()){
            ui->leTRAMGeneral->setText(file);
        }
    }
}

void MainWindow::on_cbDatLevels_toggled(bool checked)
{
    ui->leTargetDatLevels->setEnabled(checked);
}

void MainWindow::on_cbBnvLevels_toggled(bool checked)
{
    ui->leBnvLevels->setEnabled(checked);
    ui->cbGridsLevels->setEnabled(checked);
    ui->cbGridsLevels->setChecked(checked);
    if(checked){
        QString file=QFileDialog::getOpenFileName(this,"Choose the BNV.dae file...","./" , "All Files (*.*)");
        if(!file.isEmpty()){
            ui->leBnvLevels->setText(file);
        }
    }
}

void MainWindow::on_cbAdditionalSourcesLevels_toggled(bool checked)
{
    ui->leAdditSourcesLevels->setEnabled(checked);

    if(checked){
        QStringList filesSelected=QFileDialog::getOpenFileNames(this,"Choose the additional .dae files...","./" , "All Files (*.*)");
        QString filesJoined;
        int size=filesSelected.size();

        if(!filesSelected.isEmpty()){
            for(int i=0; i<size-1; i++){
                filesJoined+=filesSelected.at(i)+" ";
            }
            filesJoined+=filesSelected.at(size-1); //last doesn't have space after
            ui->leAdditSourcesLevels->setText(filesJoined);
        }

    }
}

void MainWindow::on_actionCheck_OniSplit_version_triggered()
{
    QProcess *myProcess = new QProcess();
    myProcess->start(GlobalVars::OniSplitExeName+" -version");
    myProcess->waitForFinished(-1);
    QString result=myProcess->readAllStandardOutput();
    delete myProcess;
    Util::showPopUp("This Vago version was built with base in OniSplit version "+GlobalVars::BuiltOniSplitVersion+"\n\nActual version is:\n"+result);
}

void MainWindow::on_actionCheck_xmlTools_version_triggered()
{
    QProcess *myProcess = new QProcess();
    myProcess->start(GlobalVars::XmlToolsExeName+" version");
    myProcess->waitForFinished(-1);
    QString result=myProcess->readLine();
    delete myProcess;
    Util::showPopUp("This Vago version was built with base in xmlTools version "+GlobalVars::BuiltXmlToolsVersion+"\n\nActual version is:\n"+result);
}

/**
  Update items loaded
  **/
void MainWindow::on_tabWidget_selected(const QString &arg1)
{
    if(arg1.compare("General",Qt::CaseSensitive)==0){ //case sentive is faster
        updateItemsLoaded(ui->twSourcesGeneral);
    }
    else if(arg1.compare("Textures",Qt::CaseSensitive)==0){
        updateItemsLoaded(ui->twSourcesTextures);
    }
    else if(arg1.compare("Models",Qt::CaseSensitive)==0){
        updateItemsLoaded(ui->twSourcesModels);
    }
    else if(arg1.compare("Levels",Qt::CaseSensitive)==0){
        updateItemsLoaded(ui->twSourcesLevels);
    }
    else{
        updateItemsLoaded(ui->twSourcesMisc);
    }
}

void MainWindow::updateItemsLoaded(DropTableWidget *currentTable){

    int numItems=currentTable->rowCount();

    this->itemsLoaded->setText(QString().setNum(numItems)+ (numItems==1?" item ":" items ") +"loaded");
}

void MainWindow::on_tbCommand_clicked()
{
    //Show preferences
    ManualCommands *commandsWindow = new ManualCommands(this);
    commandsWindow->show(); //it destroys itself when finished.
}

void MainWindow::on_actionWorkspace_triggered()
{
    ui->actionWorkspace->setChecked(true);
    ui->actionOther->setChecked(false);
    this->outputFolder=this->workspaceLocation;
    ui->tbOpenFolder->setToolTip("Open Vago workspace");
    showSuccessStatusMessage("Vago is now outputting the NEW items for Vago workspace.");
}

void MainWindow::on_actionOther_triggered()
{
    QString newDir=QFileDialog::getExistingDirectory(this,"Choose the folder for output NEW files directly...",this->AeLocation+"/GameDataFolder");
    newDir=Util::normalizePath(newDir);

    if(newDir.isEmpty()){
        ui->actionOther->setChecked(false);
        return; //do nothing
    }

    if(newDir==this->workspaceLocation){
        on_actionWorkspace_triggered(); //set it to vago workspace
        return;
    }

    ui->actionOther->setChecked(true);
    ui->actionWorkspace->setChecked(false);

    this->outputFolder=newDir;

    QString newDirName=Util::cutName(newDir);
    ui->tbOpenFolder->setToolTip("Open "+newDirName+" output folder");
    showSuccessStatusMessage("Vago is now outputting the NEW items for "+newDirName+".");
}

void MainWindow::on_actionView_log_triggered()
{
    Util::openLogFile();
}

QString MainWindow::getTypeConversion(DropTableWidget *myTable){
    QString from,to;

    if(myTable==ui->twSourcesGeneral){
        from=ui->cbFromGeneral->currentText();
        to=ui->cbToGeneral->currentText();
    }
    else if(myTable==ui->twSourcesTextures){
        from=ui->cbFromTextures->currentText();
        to=ui->cbToTextures->currentText();
    }
    else if(myTable==ui->twSourcesModels){
        from=ui->cbFromModels->currentText();
        to=ui->cbToModels->currentText();
    }
    else if(myTable==ui->twSourcesAnimations){
        from=ui->cbFromAnimations->currentText();
        to=ui->cbToAnimations->currentText();
    }
    else if(myTable==ui->twSourcesLevels){
        from=ui->cbFromLevels->currentText();
        to=ui->cbToLevels->currentText();
    }
    else{
        from=ui->cbFromMisc->currentText();
        to=ui->cbToMisc->currentText();
    }

    return from + " > " + to;
}

//Drop table widget context menu
void MainWindow::dtContextMenu(DropTableWidget* myTable, QContextMenuEvent *event){
    QModelIndex index = myTable->indexAt(event->pos());

    //item exists?
    if(!index.isValid())
        return;

    if(myTable->selectionModel()->selectedRows().size()==0){ //No multiple rows selected
        myTable->selectRow(myTable->itemAt(event->pos())->row()); //select all the row of the item clicked
    }

    QList<int> selectedRows = QList<int>();

    foreach(QModelIndex rowItem, myTable->selectionModel()->selectedRows()){
        selectedRows << rowItem.row();
    }

    QMenu *menu = new QMenu();
    QAction *copy = new QAction("Copy",myTable);
    QAction *moveUp = new QAction("Move Up",myTable);
    QAction *moveDown = new QAction("Move Down",myTable);
    QAction *changeOptions = new QAction("Change To Current Options",myTable);
    QMenu *changeOutput = new QMenu("Change Output for:");
    QAction *outWorkspace = new QAction("Workspace",myTable);
    QAction *outCurrOutput = new QAction("Current Output Folder",myTable);
    QAction *outOther = new QAction("Other...",myTable);
    QAction *edisable = new QAction("Enable/Disable",myTable);

    menu->addAction(copy);
    menu->addSeparator();
    menu->addAction(moveUp);
    menu->addAction(moveDown);
    menu->addSeparator();
    menu->addAction(changeOptions);
    menu->addMenu(changeOutput);
    changeOutput->addActions(QList<QAction*>() << outWorkspace << outCurrOutput << outOther);
    menu->addAction(edisable);


    //if it's in the first row it can't be setted up
    if(selectedRows.at(0)==0){
        moveUp->setEnabled(false);
    }

    //if we are at bottom we can't go down
    if(selectedRows.at(selectedRows.size()-1)==myTable->rowCount()-1){
        moveDown->setEnabled(false);
    }

    //Can we change the settings? (the conversion must be the same)
    QString currentSettings = (getTypeConversion(myTable)); //call function at the mainWindow with a signal (different threads?)
    foreach(int row, selectedRows){
        if( myTable->item(row,1)->text() != currentSettings){ //If we find out any of the selected items can't be convert disable operation
            changeOptions->setEnabled(false);
            break;
        }
    }

    QAction* selectedOption = menu->exec(event->globalPos());

    if(selectedOption==copy){
        //Let's copy the contents to the clipboard

        QString toCopy;

        int size=selectedRows.size();

        //Let's format it a bit...
        for(int i=0; i<size; i++){
            for(int j=0; j<myTable->columnCount(); j++){
                toCopy+=myTable->item(selectedRows.at(i),j)->text();
                if(j!=myTable->columnCount()-1){
                    toCopy+="\t";
                }
            }
            if(i!=size-1){
                toCopy+="\n";
            }
        }

        QApplication::clipboard()->setText(toCopy);
        showSuccessStatusMessage(QString::number(size) + (size==1?" item ":" items ")+ "copied to the clipboard");
    }
    else if(selectedOption==moveUp){
        qSort(selectedRows); //let's order the selections by the row number, so we know exactly how to swap it
        myTable->swapPositions(selectedRows,-1);
    }
    else if(selectedOption==moveDown){
        qSort(selectedRows);
        myTable->swapPositions(selectedRows,+1);
    }
    else if(selectedOption==changeOptions){
        changeToCurrentSettings(selectedRows,myTable);
    }
    else if(selectedOption==outWorkspace){
        changeItemsOutput(myTable,selectedRows,this->workspaceLocation);
    }
    else if(selectedOption==outCurrOutput){
        changeItemsOutput(myTable,selectedRows,this->outputFolder);
    }
    else if(selectedOption==outOther){

        QString newDir=QFileDialog::getExistingDirectory(this,"Choose the folder for the output of the files selected...",this->AeLocation+"/GameDataFolder");
        newDir=Util::normalizePath(newDir);

        if(newDir.isEmpty()){
            return; //do nothing
        }

        changeItemsOutput(myTable,selectedRows,newDir);

    }
    else if(selectedOption==edisable){

        int enabledCount=0, disabledCount=0;

        for(int i=0; i<selectedRows.size(); i++){

            for(int j=0; j<myTable->columnCount(); j++){
                QTableWidgetItem *currentItem=myTable->item(selectedRows.at(i),j);

                if(currentItem->background()!=myTable->disabledBackStyle){
                    myTable->setDisableStyleWidgetItem(currentItem);
                    if(j==0){ //Only count the row, not the columns
                        disabledCount++;
                    }
                }
                else{ //reset to initial state (enable)
                    myTable->resetStyleWidgetItem(currentItem);
                    if(j==0){
                        enabledCount++;
                    }
                }
            }
        }

        QString result;

        if(enabledCount!=0){
            result+=QString::number(enabledCount) + (enabledCount==1?" item ":" items ") + "Enabled";
        }
        if(enabledCount!=0 && disabledCount!=0){
            result+=" and ";
        }
        if(disabledCount!=0){
            result+=QString::number(disabledCount) + (disabledCount==1?" item ":" items ") + "Disabled";
        }

        showSuccessStatusMessage(result);
    }

    delete copy;
    delete moveUp;
    delete moveDown;
    delete changeOptions;
    delete outWorkspace;
    delete outCurrOutput;
    delete outOther;
    delete changeOutput;
    delete edisable;
    delete menu;
}

void MainWindow::changeToCurrentSettings(QList<int> rows, DropTableWidget* myTable){
    //construct a command for each one
    //Output a status message saying the number of changed files
    QString fromTo=getTypeConversion(myTable);
    QString from = QString(fromTo).remove(fromTo.indexOf(" >"),fromTo.size()-1); //parse the string to get the from, only 1 time parsed by each group of files = very fast
    QString to = QString(fromTo).remove(0,fromTo.lastIndexOf("> ")+2); //+2 to start after "> "

    QString command;

    foreach(int row, rows){

        command=getCommand(myTable,getFileOutputFolder(fromTo,myTable->getOutputAbsolute(row)),from,to,myTable->getFileAbsolute(row));

        if(command.isEmpty()){ //something wrong was happening (not inputted a texture name?)
            return; //stop changing settings
        }

        myTable->item(row,2)->setText(command); //update settings to the current row
        myTable->updateTableToolTips(row);
    }

    showSuccessStatusMessage(QString::number(rows.size()) + (rows.size()==1?" item ":" items ")+ "changed to the current settings");
}

void MainWindow::changeItemsOutput(DropTableWidget* myTable, QList<int> rows, QString newOutput){

    QString command, currentAbsoluteFile, fromTo, from, to;

    foreach(int row, rows){ //No optimization possible here, commands may be different
        fromTo=myTable->item(row,1)->text();
        from = QString(fromTo).remove(fromTo.indexOf(" >"),fromTo.size()-1); //parse the string to get the from
        to = QString(fromTo).remove(0,fromTo.lastIndexOf("> ")+2); //+2 to start after "> "

        currentAbsoluteFile=myTable->getFileAbsolute(row);
        command=getCommand(myTable,getFileOutputFolder(fromTo,newOutput),from,to,currentAbsoluteFile);

        if(command.isEmpty()){ //something wrong was happening (not inputted a texture name?)
            return; //stop changing output
        }

        myTable->item(row,2)->setText(command); //update command to the current row
        myTable->updateTableToolTips(row);
    }

    showSuccessStatusMessage(QString::number(rows.size()) + (rows.size()==1?" item ":" items ")+ "changed the output to "+(newOutput!=this->workspaceLocation?Util::cutName(newOutput):"Vago workspace"));
}

QString MainWindow::getCommand(DropTableWidget* myTable, QString myOutputFolder, QString from, QString to , QString file){
    if(myTable==ui->twSourcesGeneral){ //So we only need to parse one command.
        return fileParsingGeneral(myOutputFolder,from,to,file);
    }
    else if(myTable==ui->twSourcesTextures){
        return fileParsingTextures(myOutputFolder,from,to,file);
    }
    else if(myTable==ui->twSourcesModels){
        return fileParsingModels(myOutputFolder,from,to,file);
    }
    else if(myTable==ui->twSourcesAnimations){
        return fileParsingAnimations(myOutputFolder,from,to,file);
    }
    else if(myTable==ui->twSourcesLevels){
        return fileParsingLevels(myOutputFolder,from,to,file);
    }
    else{
        return fileParsingMisc(myOutputFolder,from,to,file);
    }

}

/**
  This is OS indepented. It maintain size ratio over the Windows and Mac.
  **/
void MainWindow::setConverterButtonsSize(){
    int height=ui->pbConvertGeneral->sizeHint().height()*1.3;
    ui->pbConvertGeneral->setMinimumHeight(height);
    ui->pbConvertTextures->setMinimumHeight(height);
    ui->pbConvertModels->setMinimumHeight(height);
    ui->pbConvertAnimations->setMinimumHeight(height);
    ui->pbConvertLevels->setMinimumHeight(height);
    ui->pbConvertMisc->setMinimumHeight(height);
}

/**
  Gets application directory. In mac os gets the .app directory
  **/
QString MainWindow::getOSIndependentAppPath(){
#ifdef Q_WS_MAC
    QDir dir = QDir::current();
    if(dir.absolutePath().contains(".app")){ // include bundle, but we don't want it
        dir.cdUp();
        dir.cdUp();
    }
    return dir.absolutePath();
#else
    return  QDir::currentPath();
#endif
}

void MainWindow::connectSlots(){

    //This signal is for thread that is working setup the progress bar (make it visible and set it's min-max)
    connect(myConverter, SIGNAL(setupPB(int)), this, SLOT(TsetupProgressBar(int)), Qt::BlockingQueuedConnection);

    //This signal is for thread that is working can update the progress bar of the gui
    connect(myConverter, SIGNAL(taskDone()), this, SLOT(TupdateProgressBar()),Qt::BlockingQueuedConnection);

    //This signal is for thread that is working can show the result of a conversion
    connect(myConverter, SIGNAL(resultConversion(QString,int)), this, SLOT(TresultConversion(QString,int)));

    //Drop signal for General table
    connect(ui->twSourcesGeneral, SIGNAL(dropped(DropTableWidget*,QStringList)), this, SLOT(addFilesSource(DropTableWidget*,QStringList)));

    //Drop signal for Textures table
    connect(ui->twSourcesTextures, SIGNAL(dropped(DropTableWidget*,QStringList)), this, SLOT(addFilesSource(DropTableWidget*,QStringList)));

    //Drop signal for Models table
    connect(ui->twSourcesModels, SIGNAL(dropped(DropTableWidget*,QStringList)), this, SLOT(addFilesSource(DropTableWidget*,QStringList)));

    //Drop signal for Animations table
    connect(ui->twSourcesAnimations, SIGNAL(dropped(DropTableWidget*,QStringList)), this, SLOT(addFilesSource(DropTableWidget*,QStringList)));

    //Drop signal for Levels table
    connect(ui->twSourcesLevels, SIGNAL(dropped(DropTableWidget*,QStringList)), this, SLOT(addFilesSource(DropTableWidget*,QStringList)));

    //Drop signal for Misc table
    connect(ui->twSourcesMisc, SIGNAL(dropped(DropTableWidget*,QStringList)), this, SLOT(addFilesSource(DropTableWidget*,QStringList)));

    //Context menu for General table
    connect(ui->twSourcesGeneral, SIGNAL(dtContextMenu(DropTableWidget*,QContextMenuEvent*)), this, SLOT(dtContextMenu(DropTableWidget*,QContextMenuEvent*)));

    //Context menu for Textures table
    connect(ui->twSourcesTextures, SIGNAL(dtContextMenu(DropTableWidget*,QContextMenuEvent*)), this, SLOT(dtContextMenu(DropTableWidget*,QContextMenuEvent*)));

    //Context menu for Models table
    connect(ui->twSourcesModels, SIGNAL(dtContextMenu(DropTableWidget*,QContextMenuEvent*)), this, SLOT(dtContextMenu(DropTableWidget*,QContextMenuEvent*)));

    //Context menu for Animations table
    connect(ui->twSourcesAnimations, SIGNAL(dtContextMenu(DropTableWidget*,QContextMenuEvent*)), this, SLOT(dtContextMenu(DropTableWidget*,QContextMenuEvent*)));

    //Context menu for Levels table
    connect(ui->twSourcesLevels, SIGNAL(dtContextMenu(DropTableWidget*,QContextMenuEvent*)), this, SLOT(dtContextMenu(DropTableWidget*,QContextMenuEvent*)));

    //Context menu for Misc table
    connect(ui->twSourcesMisc, SIGNAL(dtContextMenu(DropTableWidget*,QContextMenuEvent*)), this, SLOT(dtContextMenu(DropTableWidget*,QContextMenuEvent*)));
}


