package net.oni2.aeinstaller.backend.oni.management.tools;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.HashSet;
import java.util.TreeMap;

import net.oni2.aeinstaller.backend.Paths;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.io.xml.StaxDriver;

/**
 * @author Christian Illy
 */
public class ToolInstallationList {

	private static ToolInstallationList instance = loadCurrentList();

	private TreeMap<Integer, HashSet<String>> tools = new TreeMap<Integer, HashSet<String>>();
	private transient HashSet<Integer> modifiedPackages = new HashSet<Integer>();

	/**
	 * Add tool to entry list
	 * 
	 * @param packageId
	 *            Id of package
	 * @param files
	 *            Files of package
	 */
	public void addTool(int packageId, HashSet<String> files) {
		tools.put(packageId, files);
	}

	/**
	 * Remove tool from entry list
	 * 
	 * @param packageId
	 *            Id of package
	 */
	public void removeTool(int packageId) {
		tools.remove(packageId);
	}

	/**
	 * Check if given package id is installed as tool
	 * 
	 * @param packageId
	 *            Id of tool package
	 * @return Is installed?
	 */
	public boolean isInstalled(int packageId) {
		return tools.containsKey(packageId);
	}

	/**
	 * Retrieve installed files for package
	 * 
	 * @param packageId
	 *            Id of tool package
	 * @return File hashset
	 */
	public HashSet<String> getFiles(int packageId) {
		return tools.get(packageId);
	}

	/**
	 * @return Tool list
	 */
	public TreeMap<Integer, HashSet<String>> getItems() {
		return tools;
	}

	/**
	 * Mark a package as locally modified
	 * 
	 * @param packageId
	 *            Id of tool package
	 * @param modified
	 *            Is modified?
	 */
	public void markModified(int packageId, boolean modified) {
		if (modified)
			modifiedPackages.add(packageId);
		else
			modifiedPackages.remove(packageId);
	}

	/**
	 * @return List of all modified tools
	 */
	public HashSet<Integer> getModifiedTools() {
		return modifiedPackages;
	}

	/**
	 * Check if a specific tool is modified
	 * 
	 * @param packageId
	 *            Id of tool package
	 * @return Is tool modified?
	 */
	public boolean isModified(int packageId) {
		return modifiedPackages.contains(packageId);
	}

	/**
	 * @return Get singleton instance
	 */
	public static ToolInstallationList getInstance() {
		return instance;
	}

	private static File getFile() {
		return new File(Paths.getInstallerPath(), "installed_tools.xml");
	}

	private static XStream getStream() {
		XStream xs = new XStream(new StaxDriver());
		xs.alias("TIL", ToolInstallationList.class);
		return xs;
	}

	private static ToolInstallationList loadCurrentList() {
		ToolInstallationList til = new ToolInstallationList();
		try {
			if (getFile().exists()) {
				FileInputStream fis = new FileInputStream(getFile());
				XStream xs = getStream();
				Object obj = xs.fromXML(fis);
				if (obj instanceof ToolInstallationList) {
					til = (ToolInstallationList) obj;
					til.modifiedPackages = new HashSet<Integer>();
				}
				fis.close();
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return til;
	}

	/**
	 * Write this tool installation list to the installed_tools.xml file
	 */
	public void saveList() {
		try {
			FileOutputStream fos = new FileOutputStream(getFile());
			XStream xs = getStream();
			xs.toXML(this, fos);
			fos.close();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
}