unit Helper_LevelDB;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ComCtrls, StdCtrls, StrUtils;

type
  TForm_LevelDB = class(TForm)
    group_progress: TGroupBox;
    progress:     TProgressBar;
    lbl_progress: TLabel;
    btn_abortok:  TButton;
    lbl_estimation: TLabel;
    procedure btn_abortokClick(Sender: TObject);
  private
    procedure HandleFile(ext: String; fileid: LongWord; dir_dat2db: Boolean);
    procedure stop_convert;
  public
    procedure CreateDatabase(Source, target: String);
    procedure CreateLevel(Source, target: String);
  end;


var
  Form_LevelDB: TForm_LevelDB;

implementation

{$R *.dfm}

uses ABSMain, ABSDecUtil, Main, Functions, Data,
  OniImgClass, DataStructures, OniDataClass;

type
  THandler = procedure(fileid: LongWord; dir_dat2db: Boolean);

  TConvertHandlers = record
    Ext:     String[4];
    needed:  Boolean;
    Handler: THandler;
  end;

var
  ConvertHandlers: array of TConvertHandlers;
  loaded_filename: String;
  converting: Boolean = False;
  abort:     Boolean = False;
  DataBase:  TABSDatabase;
  Query:     TABSQuery;
  MimeCoder: TStringFormat_MIME64;

var
  DatHeader:   THeader;
  FilesHeader: TFilesMap;
  NamedFilesHeader: TNamedFilesMap;
  ExtensionsHeader: TExtensionsMap;
  Stream_Body, Stream_Names: TMemoryStream;
  Stream_Dat, Stream_Raw, Stream_Sep: TFileStream;




procedure TForm_LevelDB.CreateLevel(Source, target: String);
var
  files: LongWord;

  i, j:     LongWord;
  temps, temps2: String;
  Data, rawdata: Tdata;
  absolutebegintime, begintime: Double;
  step:     Byte;
  rawlist:  TRawList;
  extlist:  TExtensionsMap;
  fileinfo: TFileInfo;
  datlinks: TDatLinks;
  OniImage: TOniImage;
  levelid:  LongWord;
  timeformat: TFormatSettings;
const
  steps: Byte = 3;




  procedure DoStep(stepname: String);
  begin
    Inc(step);
    if stepname <> 'FIN' then
      group_progress.Caption :=
        'Creating Dat (Step ' + IntToStr(step) + '/' + IntToStr(steps) + ': ' + stepname + ')'
    else
      group_progress.Caption := 'Creating Dat (FINISHED)';
  end;

begin

  //
  // FILE EXISTS CHECK FR DAT/RAW/SEP!!!
  //

  timeformat.ShortTimeFormat := 'hh:nn:ss';
  timeformat.LongTimeFormat := 'hh:nn:ss';
  timeformat.TimeSeparator := ':';

  if not CreateDataConnection(Source, ODB_ADB) then
  begin
    ShowMessage('Could not connect to .oldb-file');
    Exit;
  end;
  levelid  := OniDataConnection.LevelInfo.LevelNumber;
  levelid  := (levelid * 2) * 256 * 256 * 256 + $01;
  OniImage := TOniImage.Create;

  absolutebegintime := Time;

  Self.Visible := True;
  Form_Main.Visible := False;
  step  := 0;
  converting := True;
  abort := False;
  btn_abortok.Caption := '&Abort...';
  btn_abortok.Default := False;
  absolutebegintime := Time;

  Stream_Body  := TMemoryStream.Create;
  Stream_Names := TMemoryStream.Create;
  Stream_Dat   := TFileStream.Create(target, fmCreate);
  Stream_Raw   := TFileStream.Create(AnsiReplaceStr(target, '.dat', '.raw'), fmCreate);
  if OniDataConnection.OSisMac then
    Stream_Sep := TFileStream.Create(AnsiReplaceStr(target, '.dat', '.sep'), fmCreate);

  DoStep('Creating header');
  progress.Position      := 0;
  lbl_progress.Caption   := '';
  lbl_estimation.Caption := 'Estimated finishing time: unknown';
  Application.ProcessMessages;

  NamedFilesHeader := TOniDataADB(OniDataConnection).GetNamedFilesMap;
  extlist := OniDataConnection.GetExtendedExtensionsList;
  for i := 0 to High(DatHeader.Ident) do
    DatHeader.Ident[i] := OniDataConnection.LevelInfo.Ident[i];
  DatHeader.Files := OniDataConnection.GetFilesCount;
  DatHeader.NamedFiles := Length(NamedFilesHeader);
  DatHeader.Extensions := Length(extlist);
  DatHeader.DataAddr   := 0;
  DatHeader.DataSize   := 0;
  DatHeader.NamesAddr  := 0;
  DatHeader.NamesSize  := 0;
  for i := 0 to High(DatHeader.Ident2) do
    DatHeader.Ident2[i] := 0;
  SetLength(FilesHeader, DatHeader.Files);
  SetLength(ExtensionsHeader, DatHeader.Extensions);

  DoStep('Writing extensions-header');
  progress.Max := Length(OniDataConnection.GetExtensionsList);
  Application.ProcessMessages;

  for i := 0 to High(ExtensionsHeader) do
  begin
    ExtensionsHeader[i].Ident     := extlist[i].Ident;
    ExtensionsHeader[i].Extension := extlist[i].Extension;
    SetLength(temps, 4);
    for j := 0 to 3 do
      temps[j + 1] := ExtensionsHeader[i].Extension[3 - j];
    ExtensionsHeader[i].ExtCount :=
      Length(OniDataConnection.GetFilesList(temps, '', False, stIDAsc));
    progress.Position    := i + 1;
    lbl_progress.Caption := 'Extensions done: ' + IntToStr(i + 1) + '/' +
      IntToStr(Length(extlist));
    Application.ProcessMessages;
  end;

  DoStep('Storing files-data');
  progress.Position := 0;
  progress.Max      := DatHeader.Files;
  lbl_progress.Caption := '';
  lbl_estimation.Caption := 'Estimated finishing time: unknown';
  Application.ProcessMessages;

  begintime := Time;
  for i := 0 to DatHeader.Files - 1 do
  begin
    fileinfo := OniDataConnection.GetFileInfo(i);
    for j := 0 to 3 do
      FilesHeader[i].Extension[j] := fileinfo.Extension[4 - j];
    if fileinfo.Size > 0 then
    begin
      //        DatLinks:=;
      FilesHeader[i].DataAddr := Stream_Body.Size + 8;
      Data    := OniDataConnection.LoadDatFile(i);
      Data[4] := (levelid) and $FF;
      Data[5] := (levelid shr 8) and $FF;
      Data[6] := (levelid shr 16) and $FF;
      Data[7] := (levelid shr 24) and $FF;

      if (Pos(UpperCase(fileinfo.Extension), UpperCase(raws)) mod 4) = 1 then
      begin
        rawlist := OniDataConnection.GetRawList(i);
        if Length(rawlist) > 0 then
        begin
          for j := 0 to High(rawlist) do
          begin
            if rawlist[j].raw_size > 0 then
            begin
              if (UpperCase(fileinfo.Extension) = 'TXMP') and
                ((Data[$88] and $01) > 0) then
              begin
                OniImage.LoadFromTXMP(i);
                OniImage.GetMipMappedImage(rawdata);
                rawlist[j].raw_size := OniImage.GetImageDataSize(True);
                Data[$90] := $08;
                Data[$89] := 32;
{                  if data[$90]<>OniImage.StoreType then begin
                    data[$90]:=OniImage.StoreType;
                    data[$89]:=(data[$89] and $CF) or $20;
                  end;
}                end
              else
              begin
                SetLength(rawdata, rawlist[j].raw_size);
                OniDataConnection.LoadRawFile(i, rawlist[j].src_offset, @rawdata[0]);
              end;
              //                data[$88]:=data[$88] and $FE;

              if rawlist[j].loc_sep then
              begin
                rawlist[j].raw_addr := Stream_Sep.Size;
                Stream_Sep.Write(rawdata[0], Length(rawdata));
              end
              else
              begin
                rawlist[j].raw_addr := Stream_Raw.Size;
                Stream_Raw.Write(rawdata[0], Length(rawdata));
              end;
            end
            else
              rawlist[j].raw_addr := 0;
            Data[rawlist[j].src_offset + 0] := (rawlist[j].raw_addr) and $FF;
            Data[rawlist[j].src_offset + 1] := (rawlist[j].raw_addr shr 8) and $FF;
            Data[rawlist[j].src_offset + 2] := (rawlist[j].raw_addr shr 16) and $FF;
            Data[rawlist[j].src_offset + 3] := (rawlist[j].raw_addr shr 24) and $FF;
          end;
        end;
      end;

      Stream_Body.Write(Data[0], Length(Data));
      //
    end
    else
      FilesHeader[i].DataAddr := 0;
    if Length(fileinfo.Name) > 0 then
    begin
      FilesHeader[i].NameAddr := Stream_Names.Size;
      temps := fileinfo.Extension + fileinfo.Name + Chr(0);
      Stream_Names.Write(temps[1], Length(temps));
    end
    else
      FilesHeader[i].NameAddr := 0;
    FilesHeader[i].FileSize := fileinfo.Size;
    FilesHeader[i].FileType := fileinfo.FileType;

    if ((i mod 10) = 0) and (i >= 100) then
      lbl_estimation.Caption := 'Estimated time left: ' + TimeToStr(
        (Time - begintime) / i * (progress.Max - i + 1) * 1.1, timeformat );
    progress.Position := i + 1;
    lbl_progress.Caption := 'Files done: ' + IntToStr(i + 1) + '/' + IntToStr(progress.Max);
    Application.ProcessMessages;
  end;

  Stream_Dat.Write(DatHeader, SizeOf(DatHeader));
  for i := 0 to High(FilesHeader) do
    Stream_Dat.Write(FilesHeader[i], SizeOf(FilesHeader[i]));
  for i := 0 to High(NamedFilesHeader) do
    Stream_Dat.Write(NamedFilesHeader[i], SizeOf(NamedFilesHeader[i]));
  for i := 0 to High(ExtensionsHeader) do
    Stream_Dat.Write(ExtensionsHeader[i], SizeOf(ExtensionsHeader[i]));

  DatHeader.DataSize  := Stream_Body.Size;
  DatHeader.NamesSize := Stream_Names.Size;
  DatHeader.DataAddr  := Stream_Dat.Size;
  Stream_Body.Seek(0, soFromBeginning);
  Stream_Dat.CopyFrom(Stream_Body, Stream_Body.Size);
  DatHeader.NamesAddr := Stream_Dat.Size;
  Stream_Names.Seek(0, soFromBeginning);
  Stream_Dat.CopyFrom(Stream_Names, Stream_Names.Size);

  Stream_Dat.Seek(0, soFromBeginning);
  Stream_Dat.Write(DatHeader, SizeOf(DatHeader));

  Stream_Dat.Free;
  Stream_Body.Free;
  Stream_Names.Free;
  Stream_Raw.Free;
  if OniDataConnection.OSisMac then
    Stream_Sep.Free;

  progress.Position      := progress.Max;
  lbl_progress.Caption   := 'Files done: ' + IntToStr(progress.Max) + '/' +
    IntToStr(progress.Max);
  lbl_estimation.Caption := 'FINISHED (duration: ' + TimeToStr(Time - absolutebegintime, timeformat) + ')';

  DoStep('FIN');
  btn_abortok.Caption := '&OK';
  btn_abortok.Default := True;

  OniImage.Free;

  converting := False;

  CloseDataConnection;
end;




procedure TForm_LevelDB.HandleFile;
var
  i: Byte;
begin
  for i := 1 to Length(ConvertHandlers) do
    if UpperCase(ConvertHandlers[i].Ext) = UpperCase(ext) then
      if ConvertHandlers[i].needed then
      begin
        ConvertHandlers[i].Handler(fileid, dir_dat2db);
        Break;
      end
      else
        Break;
end;




procedure TForm_LevelDB.CreateDatabase(Source, target: String);
var
  i, j:     LongWord;
  temps, temps2: String;
  Data:     Tdata;
  absolutebegintime, begintime: Double;
  step:     Byte;
  rawlist:  TRawList;
  extlist:  TExtensionsMap;
  fileinfo: TFileInfo;
  timeformat: TFormatSettings;
const
  steps: Byte = 4;




  procedure DoStep(stepname: String);
  begin
    Inc(step);
    if stepname <> 'FIN' then
      group_progress.Caption :=
        'Creating DB (Step ' + IntToStr(step) + '/' + IntToStr(steps) + ': ' + stepname + ')'
    else
      group_progress.Caption := 'Creating DB (FINISHED)';
  end;

begin
  if not CreateDataConnection(Source, ODB_Dat) then
  begin
    ShowMessage('Could not connect to .dat-file');
    Exit;
  end
  else
  begin
    TOniDataDat(OniDataConnection).UnloadWhenUnused := False;
  end;

  timeformat.LongTimeFormat := 'hh:nn:ss';
  timeformat.ShortTimeFormat := 'hh:nn:ss';
  timeformat.TimeSeparator := ':';

  Self.Visible := True;
  Form_Main.Visible := False;
  step  := 0;
  converting := True;
  abort := False;
  btn_abortok.Caption := '&Abort...';
  btn_abortok.Default := False;
  loaded_filename := target;

  absolutebegintime := Time;

  DataBase := TABSDatabase.Create(Self);
  DataBase.DatabaseName := 'OLDB';
  DataBase.DatabaseFileName := target;
  DataBase.CreateDatabase;

  DoStep('Creating tables');
  progress.Position      := 0;
  lbl_progress.Caption   := '';
  lbl_estimation.Caption := 'Estimated finishing time: unknown';
  Application.ProcessMessages;

  Query := TABSQuery.Create(Self);
  Query.DatabaseName := 'OLDB';
  Query.SQL.Text :=
    'CREATE TABLE globals  ( id AUTOINC PRIMARY KEY, name STRING(128), value STRING(128) );';
  Query.ExecSQL;
  Query.SQL.Text :=
    'CREATE TABLE linkmap  ( id AUTOINC PRIMARY KEY, src_id INTEGER, src_link_offset INTEGER, target_id INTEGER );';
  Query.ExecSQL;
  Query.SQL.Text :=
    'CREATE TABLE rawmap  ( id AUTOINC PRIMARY KEY, src_id INTEGER, src_link_offset INTEGER, sep BOOLEAN, size INTEGER, data BLOB BlobCompressionMode 9 BlobBlockSize 1024 BlobCompressionAlgorithm ZLib );';
  //    Query.SQL.Text:='CREATE TABLE rawmap  ( id AUTOINC PRIMARY KEY, src_id INTEGER, src_link_offset INTEGER, size INTEGER, data BLOB BlobCompressionAlgorithm None );';
  Query.ExecSQL;
  Query.SQL.Text :=
    'CREATE TABLE datfiles  ( id INTEGER PRIMARY KEY, extension CHAR(4), name STRING(128), contenttype INTEGER, size INTEGER, data BLOB BlobCompressionMode 9 BlobBlockSize 1024 BlobCompressionAlgorithm ZLib );';
  //    Query.SQL.Text:='CREATE TABLE datfiles  ( id INTEGER PRIMARY KEY, extension CHAR(4), name STRING(128), contenttype INTEGER, size INTEGER, data BLOB BlobCompressionAlgorithm None );';
  Query.ExecSQL;
  Query.SQL.Text :=
    'CREATE TABLE extlist  ( id AUTOINC PRIMARY KEY, ext CHAR(4), ident CHAR(16) );';
  Query.ExecSQL;

  Query.SQL.Text := 'INSERT INTO globals (name,value) VALUES ("dbversion","' +
    dbversion + '");';
  Query.ExecSQL;
  SetLength(Data, Length(OniDataConnection.LevelInfo.Ident));
  for i := 0 to High(OniDataConnection.LevelInfo.Ident) do
    Data[i] := OniDataConnection.LevelInfo.Ident[i];
  temps := CreateHexString(Data, True);
  Query.SQL.Text := 'INSERT INTO globals (name,value) VALUES ("ident","' + temps + '");';
  Query.ExecSQL;
  Query.SQL.Text := 'INSERT INTO globals (name,value) VALUES ("lvl","' +
    IntToStr(OniDataConnection.LevelInfo.LevelNumber) + '");';
  Query.ExecSQL;
  if OniDataConnection.OSisMAC then
    Query.SQL.Text := 'INSERT INTO globals (name,value) VALUES ("os","MAC");'
  else
    Query.SQL.Text := 'INSERT INTO globals (name,value) VALUES ("os","PC");';
  Query.ExecSQL;

  DoStep('Writing extensionslist');
  progress.Max := Length(OniDataConnection.GetExtensionsList);
  Application.ProcessMessages;

  extlist := OniDataConnection.GetExtendedExtensionsList;
  for i := 0 to High(extlist) do
  begin
    SetLength(Data, Length(extlist[i].Ident));
    for j := 0 to High(extlist[i].Ident) do
      Data[j] := extlist[i].Ident[j];
    temps := CreateHexString(Data, True);
    temps2 := extlist[i].Extension[3] + extlist[i].Extension[2] +
      extlist[i].Extension[1] + extlist[i].Extension[0];
    Query.SQL.Text := 'INSERT INTO extlist (ext,ident) VALUES ("' +
      temps2 + '","' + temps + '");';
    Query.ExecSQL;
    progress.Position    := i;
    lbl_progress.Caption := 'Extensions done: ' + IntToStr(i) + '/' +
      IntToStr(Length(extlist));
    Application.ProcessMessages;
    if abort then
    begin
      stop_convert;
      Exit;
    end;
  end;
  lbl_progress.Caption := '';

  progress.Position      := 0;
  lbl_progress.Caption   := 'Files done: ' + IntToStr(0) + '/' + IntToStr(
    OniDataConnection.GetFilesCount);
  lbl_estimation.Caption := 'Estimated finishing time: unknown';

  DoStep('Loading .dat into memory');
  Application.ProcessMessages;

  progress.Max := OniDataConnection.GetFilesCount;
  begintime    := Time;
  DoStep('Writing .dat-fileslist');
  Application.ProcessMessages;

  Database.StartTransaction;
  for i := 0 to OniDataConnection.GetFilesCount - 1 do
  begin
    fileinfo := OniDataConnection.GetFileInfo(i);
    if (fileinfo.FileType and $02) = 0 then
    begin
      mimecoder := TStringFormat_MIME64.Create;
      Data      := OniDataConnection.LoadDatFile(i);
      Query.SQL.Text :=
        'INSERT INTO datfiles (id,extension,name,contenttype,size,data) VALUES (' +
        IntToStr(i) + ',"' + fileinfo.Extension + '","' + fileinfo.Name + '","' + IntToHex(
        fileinfo.FileType, 8) + '",' + IntToStr(fileinfo.Size) + ',MimeToBin("' +
        MimeCoder.StrTo(@Data[0], Length(Data)) + '") );';
      Query.ExecSQL;
      mimecoder.Free;

      rawlist := OniDataConnection.GetRawList(i);
      if Length(rawlist) > 0 then
      begin
        for j := 0 to High(rawlist) do
        begin
          if rawlist[j].raw_size > 0 then
          begin
            SetLength(Data, rawlist[j].raw_size);
            OniDataConnection.LoadRawFile(i, rawlist[j].src_offset, Data);
            mimecoder      := TStringFormat_MIME64.Create;
            Query.SQL.Text :=
              'INSERT INTO rawmap (src_id,src_link_offset,sep,size,data) VALUES (' +
              IntToStr(i) + ',' + IntToStr(rawlist[j].src_offset) + ',' + BoolToStr(
              rawlist[j].loc_sep) + ',' + IntToStr(rawlist[j].raw_size) + ',MimeToBin("' +
              MimeCoder.StrTo(@Data[0], rawlist[j].raw_size) + '") );';
            Query.ExecSQL;
            mimecoder.Free;
          end
          else
          begin
            Query.SQL.Text :=
              'INSERT INTO rawmap (src_id,src_link_offset,sep,size) VALUES (' +
              IntToStr(i) + ',' + IntToStr(rawlist[j].src_offset) + ',' + BoolToStr(rawlist[j].loc_sep) + ',0);';
            Query.ExecSQL;
          end;
        end;
      end;

      HandleFile(fileinfo.Extension, i, True);
    end
    else
    begin
      Query.SQL.Text :=
        'INSERT INTO datfiles (id,extension,name,contenttype,size) VALUES (' +
        IntToStr(i) + ',"' + fileinfo.Extension + '","' + fileinfo.Name + '","' + IntToHex(
        fileinfo.FileType, 8) + '",0);';
      Query.ExecSQL;
    end;
    if ((i mod 100) = 0) and (i > 0) then
    begin
      Database.Commit(False);
      Database.StartTransaction;
    end;
    if ((i mod 10) = 0) and (i >= 100) then
      lbl_estimation.Caption := 'Estimated time left: ' + TimeToStr(
        (Time - begintime) / i * (progress.Max - i + 1) * 1.1, timeformat );
    progress.Position := i;
    lbl_progress.Caption := 'Files done: ' + IntToStr(i) + '/' + IntToStr(progress.Max);
    Application.ProcessMessages;
    if abort then
    begin
      stop_convert;
      Exit;
    end;
  end;
  Database.Commit(False);
  progress.Position      := progress.Max;
  lbl_progress.Caption   := 'Files done: ' + IntToStr(progress.Max) + '/' +
    IntToStr(progress.Max);

  lbl_estimation.Caption := 'FINISHED (duration: ' + TimeToStr(Time - absolutebegintime, timeformat) + ')';

  DoStep('FIN');
  btn_abortok.Caption := '&OK';
  btn_abortok.Default := True;

  converting := False;

  database.Close;
  database.Free;

  CloseDataConnection;
end;




procedure TForm_LevelDB.stop_convert;
begin
  btn_abortok.Caption := '&Close';
  btn_abortok.Default := True;
  converting := False;
  lbl_estimation.Caption := 'ABORTED';
  group_progress.Caption := 'Creating DB (ABORTED)';
  DataBase.Close;
  if MessageBox(Self.Handle, PChar('Delete the unfinished DB-file?'),
    PChar('Delete file?'), MB_YESNO) = idYes then
  begin
    DeleteFile(loaded_filename);
  end;
end;




procedure TForm_LevelDB.btn_abortokClick(Sender: TObject);
begin
  if converting then
  begin
    if MessageBox(Self.Handle,
      PChar('Do you really want to cancel the convert-progress?'),
      PChar('Warning: Converting'), MB_YESNO) = idYes then
      abort := True;
  end
  else
  begin
    Self.Visible := False;
    Form_Main.Visible  := True;
  end;
end;




procedure InsertDatLinkToDB(fileid: LongWord; offset: LongWord);
var
  link: LongWord;
begin
  OniDataConnection.LoadDatFilePart(fileid, offset, 4, @link);
  if link = 0 then
    link := $FFFFFFFF
  else
    link := link div 256;
  Query.SQL.Text := 'INSERT INTO linkmap (src_id,src_link_offset,target_id) VALUES (' +
    IntToStr(fileid) + ',' + IntToStr(offset) + ',' + IntToStr(link) + ');';
  Query.ExecSQL;
end;




procedure AISA(fileid: LongWord; dir_dat2db: Boolean);
var
  packages: Word;
  i: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1E, 2, @packages);
    if packages > 0 then
    begin
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * $160 + $28);
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * $160 + $150);
    end;
  end
  else
  begin
  end;
end;




procedure AKEV(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    for i := 0 to 16 do
      InsertDatLinkToDB(fileid, $8 + i * 4);
  end
  else
  begin
  end;
end;




procedure AKOT(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    for i := 0 to 4 do
      InsertDatLinkToDB(fileid, $8 + i * 4);
  end
  else
  begin
  end;
end;




procedure CBPI(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    for i := 0 to 56 do
      InsertDatLinkToDB(fileid, $8 + i * 4);
  end
  else
  begin
  end;
end;




procedure CBPM(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    for i := 0 to 18 do
      InsertDatLinkToDB(fileid, $8 + i * 4);
  end
  else
  begin
  end;
end;




procedure CONS(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    for i := 0 to 1 do
      InsertDatLinkToDB(fileid, $24 + i * 4);
  end
  else
  begin
  end;
end;




procedure CRSA(fileid: LongWord; dir_dat2db: Boolean);
var
  packages: LongWord;
  i: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $14, 4, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 1100 + $A0);
  end
  else
  begin
  end;
end;




procedure DOOR(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $08);
    InsertDatLinkToDB(fileid, $10);
  end
  else
  begin
  end;
end;




procedure DPGE(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $40);
  end
  else
  begin
  end;
end;




procedure HPGE(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $0C);
  end
  else
  begin
  end;
end;




procedure IGHH(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $24);
    InsertDatLinkToDB(fileid, $28);
  end
  else
  begin
  end;
end;




procedure IGPA(fileid: LongWord; dir_dat2db: Boolean);
var
  links: LongWord;
  i:     LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @links);
    if links > 0 then
      for i := 0 to links - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure IGPG(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    for i := 0 to 1 do
      InsertDatLinkToDB(fileid, $1C + i * 4);
  end
  else
  begin
  end;
end;




procedure IGSA(fileid: LongWord; dir_dat2db: Boolean);
var
  links: LongWord;
  i:     LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @links);
    if links > 0 then
      for i := 0 to links - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure IMPT(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $10);
  end
  else
  begin
  end;
end;




procedure IPGE(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $0C);
  end
  else
  begin
  end;
end;




procedure KEYI(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    for i := 0 to 9 do
      InsertDatLinkToDB(fileid, $08 + i * 4);
  end
  else
  begin
  end;
end;




procedure M3GA(fileid: LongWord; dir_dat2db: Boolean);
var
  links: LongWord;
  i:     LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @links);
    if links > 0 then
      for i := 0 to links - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure M3GM(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    for i := 0 to 6 do
      InsertDatLinkToDB(fileid, $0C + i * 4);
  end
  else
  begin
  end;
end;




procedure MTRL(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $10);
  end
  else
  begin
  end;
end;




procedure OBDC(fileid: LongWord; dir_dat2db: Boolean);
var
  packages: Word;
  i: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1E, 2, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * $18 + $4);
  end
  else
  begin
  end;
end;




procedure OBOA(fileid: LongWord; dir_dat2db: Boolean);
var
  packages: Word;
  i: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1E, 2, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
      begin
        InsertDatLinkToDB(fileid, $20 + i * 240 + $0);
        InsertDatLinkToDB(fileid, $20 + i * 240 + $4);
        InsertDatLinkToDB(fileid, $20 + i * 240 + $8);
      end;
  end
  else
  begin
  end;
end;




procedure OFGA(fileid: LongWord; dir_dat2db: Boolean);
var
  packages: LongWord;
  i: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 12 + $04);
  end
  else
  begin
  end;
end;




procedure ONCC(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $28);
    InsertDatLinkToDB(fileid, $434);
    InsertDatLinkToDB(fileid, $438);
    InsertDatLinkToDB(fileid, $43C);
    InsertDatLinkToDB(fileid, $C3C);
    InsertDatLinkToDB(fileid, $C40);
    InsertDatLinkToDB(fileid, $C44);
    InsertDatLinkToDB(fileid, $C48);
    InsertDatLinkToDB(fileid, $C88);
    InsertDatLinkToDB(fileid, $C8C);
  end
  else
  begin
  end;
end;




procedure ONCV(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $08);
  end
  else
  begin
  end;
end;




procedure ONLV(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    for i := 0 to 5 do
      InsertDatLinkToDB(fileid, $48 + i * 4);
    for i := 0 to 5 do
      InsertDatLinkToDB(fileid, $64 + i * 4);
    InsertDatLinkToDB(fileid, $300);
  end
  else
  begin
  end;
end;




procedure ONOA(fileid: LongWord; dir_dat2db: Boolean);
var
  packages: LongWord;
  i: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 8 + $04);
  end
  else
  begin
  end;
end;




procedure ONSK(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $08);
    InsertDatLinkToDB(fileid, $0C);
    InsertDatLinkToDB(fileid, $10);
    InsertDatLinkToDB(fileid, $14);
    InsertDatLinkToDB(fileid, $18);
    InsertDatLinkToDB(fileid, $20);
    InsertDatLinkToDB(fileid, $44);
  end
  else
  begin
  end;
end;




procedure ONVL(fileid: LongWord; dir_dat2db: Boolean);
var
  packages: LongWord;
  i: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure ONWC(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $28);
    InsertDatLinkToDB(fileid, $34);
    InsertDatLinkToDB(fileid, $40);
    InsertDatLinkToDB(fileid, $54);
    InsertDatLinkToDB(fileid, $58);
    InsertDatLinkToDB(fileid, $5C);
    InsertDatLinkToDB(fileid, $60);
    InsertDatLinkToDB(fileid, $6FC);
    InsertDatLinkToDB(fileid, $700);
  end
  else
  begin
  end;
end;




procedure OPGE(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $0C);
  end
  else
  begin
  end;
end;




procedure PSPC(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $50);
  end
  else
  begin
  end;
end;




procedure PSPL(fileid: LongWord; dir_dat2db: Boolean);
var
  packages: LongWord;
  i: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 8 + $4);
  end
  else
  begin
  end;
end;




procedure PSUI(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    for i := 0 to 43 do
      InsertDatLinkToDB(fileid, $08 + i * 4);
  end
  else
  begin
  end;
end;




procedure STNA(fileid: LongWord; dir_dat2db: Boolean);
var
  packages: Word;
  i: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1E, 2, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure TRAC(fileid: LongWord; dir_dat2db: Boolean);
var
  packages: Word;
  i: LongWord;
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $18);
    OniDataConnection.LoadDatFilePart(fileid, $1E, 2, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 12 + 8);
  end
  else
  begin
  end;
end;




procedure TRAM(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $40);
    InsertDatLinkToDB(fileid, $44);
  end
  else
  begin
  end;
end;




procedure TRAS(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $08);
  end
  else
  begin
  end;
end;




procedure TRBS(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    for i := 0 to 4 do
      InsertDatLinkToDB(fileid, $08 + i * 4);
  end
  else
  begin
  end;
end;




procedure TRCM(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
begin
  if dir_dat2db then
  begin
    for i := 0 to 2 do
      InsertDatLinkToDB(fileid, $5C + i * 4);
  end
  else
  begin
  end;
end;




procedure TRGA(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
  packages: Word;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1E, 2, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure TRGE(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $20);
  end
  else
  begin
  end;
end;




procedure TRIG(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $18);
    InsertDatLinkToDB(fileid, $24);
    InsertDatLinkToDB(fileid, $28);
  end
  else
  begin
  end;
end;




procedure TRMA(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
  packages: Word;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1E, 2, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure TRSC(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
  packages: Word;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1E, 2, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure TSFF(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
  packages: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure TSFT(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $1C);
  end
  else
  begin
  end;
end;




procedure TURR(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $60);
    InsertDatLinkToDB(fileid, $6C);
    InsertDatLinkToDB(fileid, $74);
  end
  else
  begin
  end;
end;




procedure TXAN(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
  packages: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure TXMA(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
  packages: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure TXMB(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
  packages: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure TXMP(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $94);
    InsertDatLinkToDB(fileid, $98);
  end
  else
  begin
  end;
end;




procedure TXTC(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $08);
  end
  else
  begin
  end;
end;




procedure WMCL(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
  packages: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 8 + $4);
  end
  else
  begin
  end;
end;




procedure WMDD(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
  packages: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $11C, 4, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $120 + i * $124 + $114);
  end
  else
  begin
  end;
end;




procedure WMMB(fileid: LongWord; dir_dat2db: Boolean);
var
  i: LongWord;
  packages: LongWord;
begin
  if dir_dat2db then
  begin
    OniDataConnection.LoadDatFilePart(fileid, $1C, 4, @packages);
    if packages > 0 then
      for i := 0 to packages - 1 do
        InsertDatLinkToDB(fileid, $20 + i * 4);
  end
  else
  begin
  end;
end;




procedure WPGE(fileid: LongWord; dir_dat2db: Boolean);
begin
  if dir_dat2db then
  begin
    InsertDatLinkToDB(fileid, $08);
    InsertDatLinkToDB(fileid, $0C);
  end
  else
  begin
  end;
end;




procedure InsertHandler(ext: String; needed: Boolean; handler: THandler);
begin
  SetLength(ConvertHandlers, Length(ConvertHandlers) + 1);
  ConvertHandlers[High(ConvertHandlers)].Ext     := ext;
  ConvertHandlers[High(ConvertHandlers)].needed  := needed;
  ConvertHandlers[High(ConvertHandlers)].handler := handler;
end;

begin
  InsertHandler('ABNA', False, nil);
  //  InsertHandler('AGDB',True,AGDB);
  InsertHandler('AGDB', False, nil);
  InsertHandler('AGQC', False, nil);
  InsertHandler('AGQG', False, nil);
  InsertHandler('AGQR', False, nil);
  InsertHandler('AISA', True, AISA);
  InsertHandler('AITR', False, nil);
  InsertHandler('AKAA', False, nil);
  InsertHandler('AKBA', False, nil);
  InsertHandler('AKBP', False, nil);
  InsertHandler('AKDA', False, nil);
  InsertHandler('AKEV', True, AKEV);
  InsertHandler('AKOT', True, AKOT);
  InsertHandler('AKVA', False, nil);
  InsertHandler('BINA', False, nil);
  InsertHandler('CBPI', True, CBPI);
  InsertHandler('CBPM', True, CBPM);
  InsertHandler('CONS', True, CONS);
  InsertHandler('CRSA', True, CRSA);
  InsertHandler('DOOR', True, DOOR);
  InsertHandler('DPGE', True, DPGE);
  InsertHandler('ENVP', False, nil);
  InsertHandler('FILM', False, nil);
  InsertHandler('HPGE', True, HPGE);
  InsertHandler('IDXA', False, nil);
  InsertHandler('IGHH', True, IGHH);
  InsertHandler('IGPA', True, IGPA);
  InsertHandler('IGPG', True, IGPG);
  InsertHandler('IGSA', True, IGSA);
  InsertHandler('IMPT', True, IMPT);
  InsertHandler('IPGE', True, IPGE);
  InsertHandler('KEYI', True, KEYI);
  InsertHandler('M3GA', True, M3GA);
  InsertHandler('M3GM', True, M3GM);
  InsertHandler('MTRL', True, MTRL);
  InsertHandler('OBAN', False, nil);
  InsertHandler('OBDC', True, OBDC);
  InsertHandler('OBOA', True, OBOA);
  InsertHandler('OFGA', True, OFGA);
  InsertHandler('ONCC', True, ONCC);
  InsertHandler('ONCP', False, nil);
  InsertHandler('ONCV', True, ONCV);
  InsertHandler('ONFA', False, nil);
  InsertHandler('ONGS', False, nil);
  InsertHandler('ONIA', False, nil);
  InsertHandler('ONLD', False, nil);
  InsertHandler('ONLV', True, ONLV);
  InsertHandler('ONMA', False, nil);
  InsertHandler('ONOA', True, ONOA);
  InsertHandler('ONSA', False, nil);
  InsertHandler('ONSK', True, ONSK);
  InsertHandler('ONTA', False, nil);
  InsertHandler('ONVL', True, ONVL);
  InsertHandler('ONWC', True, ONWC);
  InsertHandler('OPGE', True, OPGE);
  InsertHandler('OSBD', False, nil);
  InsertHandler('OTIT', False, nil);
  InsertHandler('OTLF', False, nil);
  InsertHandler('PLEA', False, nil);
  InsertHandler('PNTA', False, nil);
  InsertHandler('PSPC', True, PSPC);
  InsertHandler('PSPL', True, PSPL);
  InsertHandler('PSUI', True, PSUI);
  InsertHandler('QTNA', False, nil);
  InsertHandler('SNDD', False, nil);
  InsertHandler('STNA', True, STNA);
  InsertHandler('SUBT', False, nil);
  InsertHandler('TRAC', True, TRAC);
  InsertHandler('TRAM', True, TRAM);
  InsertHandler('TRAS', True, TRAS);
  InsertHandler('TRBS', True, TRBS);
  InsertHandler('TRCM', True, TRCM);
  InsertHandler('TRGA', True, TRGA);
  InsertHandler('TRGE', True, TRGE);
  InsertHandler('TRIA', False, nil);
  InsertHandler('TRIG', True, TRIG);
  InsertHandler('TRMA', True, TRMA);
  InsertHandler('TRSC', True, TRSC);
  InsertHandler('TRTA', False, nil);
  InsertHandler('TSFF', True, TSFF);
  InsertHandler('TSFL', False, nil);
  InsertHandler('TSFT', True, TSFT);
  InsertHandler('TSGA', False, nil);
  InsertHandler('TSTR', False, nil);
  InsertHandler('TURR', True, TURR);
  InsertHandler('TXAN', True, TXAN);
  InsertHandler('TXCA', False, nil);
  InsertHandler('TXMA', True, TXMA);
  InsertHandler('TXMB', True, TXMB);
  InsertHandler('TXMP', True, TXMP);
  InsertHandler('TXTC', True, TXTC);
  InsertHandler('VCRA', False, nil);
  InsertHandler('WMCL', True, WMCL);
  InsertHandler('WMDD', True, WMDD);
  InsertHandler('WMM_', False, nil);
  InsertHandler('WMMB', True, WMMB);
  InsertHandler('WPGE', True, WPGE);
end.
