#include "xmlcustomcode.h"

// http://stackoverflow.com/questions/7092765/what-does-it-mean-to-have-an-undefined-reference-to-a-static-member
QVector<QScriptEngine*> XmlCustomCode::scriptEngines;
QVector<QScriptValue*> XmlCustomCode::jsFunctions;
QVector<QScriptValue*> XmlCustomCode::getXmlDataFunctions;
QVector<QScriptValue*> XmlCustomCode::setXmlDataFunctions;

QScriptValue echo(QScriptContext *context, QScriptEngine*)
{
    std::cout << context->argument(0).toString().toUtf8().constData() << std::endl;

    return "";
}

XmlCustomCode::XmlCustomCode(): numThreads(omp_get_num_procs()*2)
{
    // create individual thread script engines if empty
    if(this->scriptEngines.isEmpty()){
        this->scriptEngines.reserve(this->numThreads);
        this->jsFunctions.reserve(this->numThreads);
        this->getXmlDataFunctions.reserve(this->numThreads);
        this->setXmlDataFunctions.reserve(this->numThreads);

        QString jsxmlString;
        QFile jsxmlfile(":/resources/libs/jsxml.js");

        jsxmlfile.open(QFile::ReadOnly | QFile::Text);

        jsxmlString=QTextStream(&jsxmlfile).readAll();

        for(int i=0; i<this->numThreads; i++){
            this->scriptEngines.append(new QScriptEngine());
            this->jsFunctions.append(new QScriptValue());

            // main needs to be called so the user code is evaluated
            // alternatively you can do: myFunc=engine.evaluate('(function main(){})'); myFunc.call();
            // Note the () around the function
            this->getXmlDataFunctions.append(new QScriptValue(this->scriptEngines[i]->evaluate("(function getXmlData() { return $xmlData; })")));
            this->setXmlDataFunctions.append(new QScriptValue(this->scriptEngines[i]->evaluate("(function setXmlData(newXmlData) { $xmlData=newXmlData; })")));

            // Add echo function so user can debug the code
            QScriptValue echoFunction = this->scriptEngines[i]->newFunction(echo);
            this->scriptEngines[i]->globalObject().setProperty("echo", echoFunction);

            // Add the js library for XmlEditing
            this->scriptEngines[i]->evaluate(jsxmlString);
        }
    }
}

void XmlCustomCode::executeCustomCode(const QString &jsString, const QVector<QString> &filesToProcess, const bool backupsEnabled, const bool verboseEnabled){

    // Reconstruct script functions
    for(int i=0; i<this->numThreads; i++){
        *this->jsFunctions[i]=this->scriptEngines[i]->evaluate("(function main() {"+jsString+"})");
    }

    QString currXmlFileString;

    QScriptValue engineResult; // variable to check for js_errors
    double elapsed_secs=0; // elapsed seconds that a user script took
    clock_t begin; // seconds that a script started

    // Single tread if small files
    if(filesToProcess.size()<CUSTOM_FILES_PER_THREAD){
        // Process all XmlFiles
        for(int i=0; i<filesToProcess.size(); i++){

            customCodeUnwinding(filesToProcess.at(i),currXmlFileString,*this->scriptEngines[0],begin,elapsed_secs,engineResult,
                    *this->jsFunctions[0],*this->getXmlDataFunctions[0],*this->setXmlDataFunctions[0],backupsEnabled,verboseEnabled);
        }
    }
    else{ // Multithread if there are many files
        // Process all XmlFiles
#pragma omp parallel for num_threads(this->numThreads) schedule(dynamic)
        for(int i=0; i<filesToProcess.size()-CUSTOM_FILES_PER_THREAD; i+=CUSTOM_FILES_PER_THREAD){

            const int tid=omp_get_thread_num();

            QString currXmlFileStringThread;

            QScriptValue engineResultThread; // variable to check for js_errors
            double elapsedSecsThread=0; // elapsed seconds that a user script took
            clock_t beginThread; // seconds that a script started

            customCodeUnwinding(filesToProcess.at(i),currXmlFileStringThread,*this->scriptEngines[tid],beginThread,elapsedSecsThread,engineResultThread,
                                *this->jsFunctions[tid],*this->getXmlDataFunctions[tid],*this->setXmlDataFunctions[tid],backupsEnabled,verboseEnabled);

            customCodeUnwinding(filesToProcess.at(i+1),currXmlFileStringThread,*this->scriptEngines[tid],beginThread,elapsedSecsThread,engineResultThread,
                                *this->jsFunctions[tid],*this->getXmlDataFunctions[tid],*this->setXmlDataFunctions[tid],backupsEnabled,verboseEnabled);
        }

        if(filesToProcess.size()%CUSTOM_FILES_PER_THREAD!=0){

            int alreadyProcessedFiles=(filesToProcess.size()/CUSTOM_FILES_PER_THREAD)*CUSTOM_FILES_PER_THREAD;

            for(int i=alreadyProcessedFiles; i<filesToProcess.size(); i++){

                customCodeUnwinding(filesToProcess.at(i),currXmlFileString,*this->scriptEngines[0],begin,elapsed_secs,engineResult,
                        *this->jsFunctions[0],*this->getXmlDataFunctions[0],*this->setXmlDataFunctions[0],backupsEnabled,verboseEnabled);
            }
        }
    }
}

void XmlCustomCode::displayJsException(QScriptEngine &engine, QScriptValue &engineResult){
    if (engine.hasUncaughtException()) {
        UtilXmlTools::displayErrorMessage("@CUSTOM_CODE","Uncaught js exception (user code) at line " +QString::number(engine.uncaughtExceptionLineNumber()) + ":\n" + engineResult.toString());
    }
}
